/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package tlsconfig

import (
	"reflect"
	"testing"
)

func TestDetails_Validate(t *testing.T) {
	type fields struct {
		SessionName        string
		TlsConnect         string
		TlsCaFile          string
		TlsCertFile        string
		TlsKeyFile         string
		RawUri             string
		AllowedConnections map[string]bool
	}
	type args struct {
		checkCA       bool
		checkCertFile bool
		checkKeyFile  bool
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		wantErr bool
	}{
		{
			"full",
			fields{"name", "foo", "caFile", "certFile", "keyFile", "uri", map[string]bool{"foo": true, "bar": true}},
			args{true, true, true},
			false,
		},
		{
			"checkCA",
			fields{TlsConnect: "foo", TlsCaFile: "file", AllowedConnections: map[string]bool{"foo": true}},
			args{true, false, false},
			false,
		},
		{
			"checkKey",
			fields{TlsConnect: "foo", TlsKeyFile: "file", AllowedConnections: map[string]bool{"foo": true}},
			args{false, false, true},
			false,
		},
		{
			"checkCert",
			fields{TlsConnect: "foo", TlsCertFile: "file", AllowedConnections: map[string]bool{"foo": true}},
			args{false, true, false},
			false,
		},
		{
			"checkCA_fail",
			fields{TlsConnect: "foo", AllowedConnections: map[string]bool{"foo": true}},
			args{true, false, false},
			true,
		},
		{
			"checkKey_fail",
			fields{TlsConnect: "foo", AllowedConnections: map[string]bool{"foo": true}},
			args{false, false, true},
			true,
		},
		{
			"checkCert_fail",
			fields{TlsConnect: "foo", AllowedConnections: map[string]bool{"foo": true}},
			args{false, true, false},
			true,
		},
		{
			"AllowedConnections_fail",
			fields{TlsConnect: "foo"},
			args{false, true, false},
			true,
		},
		{
			"TlsConnect_fail",
			fields{AllowedConnections: map[string]bool{"foo": true}},
			args{false, true, false},
			true,
		},
		{
			"not_in_allowed_connections",
			fields{TlsConnect: "foo", AllowedConnections: map[string]bool{"bar": true}},
			args{false, true, false},
			true,
		},
		{"empty", fields{}, args{}, true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			d := &Details{
				SessionName:        tt.fields.SessionName,
				TlsConnect:         tt.fields.TlsConnect,
				TlsCaFile:          tt.fields.TlsCaFile,
				TlsCertFile:        tt.fields.TlsCertFile,
				TlsKeyFile:         tt.fields.TlsKeyFile,
				RawUri:             tt.fields.RawUri,
				AllowedConnections: tt.fields.AllowedConnections,
			}
			if err := d.Validate(tt.args.checkCA, tt.args.checkCertFile, tt.args.checkKeyFile); (err != nil) != tt.wantErr {
				t.Errorf("Details.Validate() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestNewDetails(t *testing.T) {
	type args struct {
		session         string
		dbConnect       string
		caFile          string
		certFile        string
		keyFile         string
		uri             string
		connectionTypes []string
	}
	tests := []struct {
		name string
		args args
		want Details
	}{
		{
			"default",
			args{"name", "conn", "caFile", "certFile", "keyFile", "uri", nil},
			Details{"name", "conn", "caFile", "certFile", "keyFile", "uri", map[string]bool{}},
		},
		{
			"with_conn_types",
			args{"name", "conn", "caFile", "certFile", "keyFile", "uri", []string{"foo", "bar"}},
			Details{"name", "conn", "caFile", "certFile", "keyFile", "uri", map[string]bool{"foo": true, "bar": true}},
		},
		{"empty", args{}, Details{AllowedConnections: map[string]bool{}}},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := NewDetails(tt.args.session, tt.args.dbConnect, tt.args.caFile, tt.args.certFile, tt.args.keyFile, tt.args.uri, tt.args.connectionTypes...); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("NewDetails() = %v, want %v", got, tt.want)
			}
		})
	}
}
