// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief B(*) bar{B}(*) exclusive cross section
  class BELLE_2024_I2791218 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BELLE_2024_I2791218);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      // histograms
      book(_sigma[0], 2, 1, 1);
      book(_sigma[1], 2, 1, 2);
      book(_sigma[2], 2, 1, 3);

      // Check which indices match the Ecm energy
      for (const auto& est : refData<YODA::BinnedEstimate<int>>(1, 1, 1).bins()) {
        if (abs(sqrtS()/MeV-est.val())<est.totalErrAvg()) _sqs.push_back(est.xEdge());
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle& child : p.children()) {
        if (child.children().empty()) {
          nRes[child.pid()] -= 1;
          --ncount;
        }
        else findChildren(child,nRes,ncount);
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      // extract botton hadrons
      Particles bHadrons=apply<FinalState>(event, "UFS").particles(Cuts::abspid==511 ||
                                                                   Cuts::abspid==513 ||
                                                                   Cuts::abspid==521 ||
                                                                   Cuts::abspid==523);
      for (size_t ix=0; ix<bHadrons.size(); ++ix) {
        long pix = bHadrons[ix].parents()[0].abspid();
        if (pix==511 || pix==413 || pix==521 || pix==523) continue;
        map<long,int> nRes = nCount;
        int ncount = ntotal;
        findChildren(bHadrons[ix],nRes,ncount);
        bool matched=false;
        for (size_t iy=ix+1; iy<bHadrons.size(); ++iy) {
          long piy = bHadrons[ix].parents()[0].abspid();
          if (piy==511 || piy==413 || piy==521 || piy==523) continue;
          map<long,int> nRes2 = nRes;
          int ncount2 = ncount;
          findChildren(bHadrons[iy],nRes2,ncount2);
          if (ncount2!=0) continue;
          matched=true;
          for (const auto& val : nRes2) {
            if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) {
            for (const int Ecm : _sqs) {
              if (bHadrons[ix].abspid()==511 || bHadrons[ix].abspid()==521) {
                if (bHadrons[iy].pid()==-bHadrons[ix].pid()) _sigma[0]->fill(Ecm);
                else                                         _sigma[1]->fill(Ecm);
              }
              else if (bHadrons[ix].abspid()==513 || bHadrons[ix].abspid()==523) {
                if (bHadrons[iy].pid()==-bHadrons[ix].pid()) _sigma[2]->fill(Ecm);
                else                                         _sigma[1]->fill(Ecm);
              }
            }
            break;
          }
        }
        if (matched) break;
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_sigma, crossSection()/sumOfWeights()/picobarn);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<int> _sigma[3];
    vector<int> _sqs;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BELLE_2024_I2791218);
}
