// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/devtools/device/usb/android_rsa.h"

#include <array>
#include <optional>
#include <string>
#include <string_view>

#include "base/strings/string_view_util.h"
#include "crypto/hash.h"
#include "crypto/keypair.h"
#include "crypto/rsa_private_key.h"
#include "crypto/sign.h"
#include "crypto/test_support.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace {

TEST(AndroidRSATest, EncodePublicKey) {
  auto key = crypto::test::FixedRsa2048PrivateKeyForTesting();
  std::string_view expected =
      "QAAAAIuxyjLdkf9dwbd1cL9LyHJ2RPiBOob18Y8buAkZfERfSAQ2N7dTCc763pGJ2VyBj3kZ"
      "b7uhsG6KQz9RyY8dOJxX4ihd97sEEju6y277D2vsjP1WhSPHomOw19yxH3YKQOE0z6/3TeCF"
      "VVvW+7t2wc4LLF5xnrHvhGwI1YftkJcXU73OOLmHKOnd1uPFleO7IDj/Na41qsaEjoCFn4XY"
      "Z9NAUY6W8ws/ge8BV1gaD0QibRnBAylUjf6D43p3RlOo1QOubtI2YqDm+w3wopIS0LkesctP"
      "AsShaNyTndv3xd8gNCcHvTAupjZ3fNhd9dpxrFQ/wnBtACQXwYGrRd+4VkehEXO3LPo3yhS5"
      "wD0VAXGg/ah+fSICe+zEt8pb3Ivy26v1eprkO4/dPwe4Mb15HGOmKJxHtnIWThOZgb9/dUnc"
      "W9CQ1sldHAQuRAofrK2uXsVFksOu6oU8891WCpkoSdQOmi47thhTBQiC4v7ADUwz0EVnHpxD"
      "eGChiZTqv1xIZlL+LoDeymZJQ6xul3Ipgu94Z5WqqOUSu/nzhbzcbQMdiB727n4ZM+INSxmd"
      "F5l0GR6DzDb/nF52MZjPgNCHmjEa6vM/mkEjAFZVz7f4VfSg+MgWm9iO86YvfG+QyoZToSzt"
      "dZa5hBvzNQk/jJHYa7EGrCq1UOa4ctVeb6Nig29n3cgxfAEAAQA=";
  EXPECT_EQ(AndroidRSAPublicKey(key), expected);
}

TEST(AndroidRSATest, EncodePublicKeyTooSmall) {
  constexpr auto kTestRSA1024Key = std::to_array<uint8_t>(
      {0x30, 0x82, 0x02, 0x76, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
       0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
       0x02, 0x60, 0x30, 0x82, 0x02, 0x5c, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81,
       0x00, 0xcf, 0xee, 0xba, 0xe3, 0x82, 0x9e, 0x6d, 0x59, 0x82, 0x9c, 0xd7,
       0xf0, 0x0b, 0x77, 0x73, 0x51, 0xb5, 0xf8, 0x7e, 0xf2, 0x90, 0x0e, 0x4e,
       0xe7, 0x3a, 0xa4, 0x7b, 0x2b, 0x69, 0x24, 0x16, 0x07, 0x3d, 0x48, 0x68,
       0x4f, 0x5f, 0x70, 0x35, 0x28, 0x47, 0x5b, 0xb1, 0xc0, 0xb7, 0xfc, 0x29,
       0x36, 0x67, 0x81, 0x12, 0xc4, 0xe4, 0x7f, 0x53, 0x63, 0x42, 0x0c, 0xe4,
       0x45, 0x09, 0xee, 0xcd, 0x83, 0x33, 0xb6, 0x33, 0x20, 0xf4, 0xf7, 0x03,
       0x2c, 0x42, 0xbf, 0x2a, 0xd7, 0x51, 0x1e, 0x5b, 0x28, 0x35, 0xe8, 0xe3,
       0x35, 0x63, 0x62, 0x9f, 0x2f, 0x84, 0xcf, 0xb7, 0xfd, 0x46, 0xbf, 0x0b,
       0xbb, 0xb3, 0xd8, 0x1f, 0xdb, 0x4e, 0xae, 0x46, 0x9c, 0x87, 0x94, 0xbe,
       0xcf, 0x24, 0x92, 0xfe, 0xf8, 0x72, 0xc6, 0x16, 0xc2, 0xd2, 0x31, 0x52,
       0x21, 0x91, 0xc1, 0x4a, 0xa5, 0x5a, 0x43, 0x6d, 0xcf, 0x02, 0x03, 0x01,
       0x00, 0x01, 0x02, 0x81, 0x80, 0x14, 0xb2, 0xf3, 0x2f, 0xca, 0xc4, 0xf2,
       0xa5, 0xc2, 0x76, 0xd2, 0x98, 0x36, 0xcc, 0x01, 0x52, 0xd5, 0x06, 0x93,
       0xbc, 0x99, 0xd2, 0x92, 0xd0, 0x39, 0xea, 0x3d, 0xc8, 0xa8, 0x82, 0xcb,
       0x1e, 0x95, 0x5f, 0x2b, 0xe3, 0x1a, 0xe1, 0x14, 0xa9, 0x99, 0xd0, 0xea,
       0xf5, 0x00, 0x5b, 0xbd, 0xce, 0x88, 0xa7, 0x34, 0xb3, 0x1c, 0xfb, 0xff,
       0xd5, 0x15, 0xb2, 0x2a, 0xd7, 0x24, 0x91, 0xfd, 0x2b, 0x30, 0x5a, 0x43,
       0xa1, 0x36, 0xd5, 0xb5, 0x64, 0x4d, 0xd1, 0x11, 0x4a, 0x19, 0xf9, 0x0a,
       0x3a, 0x21, 0x33, 0x35, 0x87, 0xfb, 0xdd, 0xf9, 0xa6, 0x3c, 0xd1, 0x51,
       0xc8, 0x2f, 0x28, 0x27, 0x20, 0x75, 0xbd, 0x56, 0x2b, 0xb9, 0xff, 0x23,
       0x31, 0x4b, 0xcb, 0x42, 0x1f, 0x01, 0x1b, 0xea, 0x93, 0x53, 0xf4, 0x95,
       0xae, 0x8e, 0x6f, 0x0f, 0xb0, 0xf1, 0x11, 0x67, 0x02, 0x5f, 0xe5, 0x9e,
       0xe1, 0x02, 0x41, 0x00, 0xf9, 0xc1, 0xc6, 0x6f, 0x26, 0xa0, 0xb0, 0x0d,
       0x35, 0x37, 0xa6, 0xf6, 0xd8, 0xcf, 0x35, 0xef, 0x8e, 0x32, 0xce, 0x65,
       0x24, 0x0a, 0x2d, 0x32, 0xd8, 0xe6, 0x4d, 0xe2, 0x1f, 0xec, 0x3f, 0xed,
       0xd2, 0x13, 0x79, 0x61, 0x2e, 0x55, 0x1a, 0xc0, 0x10, 0x50, 0xd2, 0x1c,
       0x54, 0x09, 0xbf, 0x6b, 0xd7, 0x2c, 0xeb, 0xf8, 0x95, 0x8f, 0x88, 0xfc,
       0x8c, 0x2a, 0x04, 0x4e, 0x54, 0x08, 0x31, 0x23, 0x02, 0x41, 0x00, 0xd5,
       0x21, 0x50, 0xc6, 0xbd, 0xe6, 0x17, 0x15, 0x82, 0x68, 0xf1, 0x77, 0x7e,
       0x4e, 0xaa, 0xab, 0x41, 0xf3, 0x07, 0xf9, 0x79, 0x07, 0x8d, 0x4a, 0x1c,
       0xc8, 0x59, 0xe1, 0x5a, 0x9b, 0x4c, 0xcc, 0xe7, 0xf9, 0xdc, 0xe6, 0x6b,
       0xe3, 0xe0, 0x58, 0xd1, 0x2e, 0x58, 0x98, 0xf2, 0x53, 0x1c, 0x1b, 0x2f,
       0x24, 0xa3, 0xae, 0x9b, 0x87, 0x55, 0x4f, 0x0e, 0x9a, 0x41, 0x03, 0x18,
       0x1f, 0xf9, 0x65, 0x02, 0x41, 0x00, 0xe7, 0x40, 0x45, 0x44, 0x9b, 0x66,
       0x83, 0x3b, 0xe7, 0x7f, 0x7d, 0x07, 0xe8, 0x5d, 0x8b, 0xda, 0x20, 0x36,
       0xe8, 0xa5, 0xfb, 0x45, 0xb4, 0xed, 0xf4, 0xa8, 0xe1, 0x8a, 0x32, 0xf6,
       0xa3, 0x86, 0xc7, 0x51, 0xa2, 0xbc, 0x99, 0x92, 0xd1, 0x2e, 0xee, 0xac,
       0x76, 0xaf, 0x26, 0x1c, 0xd4, 0x71, 0xbc, 0x8d, 0x6a, 0x7c, 0x03, 0xb7,
       0x20, 0x26, 0x7e, 0x7d, 0x54, 0xf9, 0x55, 0x93, 0x34, 0x33, 0x02, 0x40,
       0x68, 0xd0, 0x57, 0x8f, 0x45, 0xb8, 0xb0, 0x45, 0x8d, 0xba, 0xda, 0x75,
       0x6f, 0x92, 0x74, 0x70, 0x80, 0xa0, 0x6a, 0x9d, 0x8f, 0x9d, 0xdd, 0x28,
       0x43, 0x7b, 0x6c, 0x68, 0x5d, 0x4c, 0xde, 0xba, 0x2e, 0xa3, 0x3c, 0xb5,
       0xbc, 0xfd, 0x76, 0x49, 0x77, 0xa6, 0x19, 0x40, 0x2e, 0x7c, 0x1c, 0xe8,
       0x69, 0x88, 0xd5, 0x4e, 0x7b, 0x82, 0x54, 0xbb, 0xea, 0x6c, 0x85, 0xe8,
       0x65, 0xa6, 0xe3, 0xd9, 0x02, 0x40, 0x06, 0x07, 0xb2, 0x3a, 0xde, 0xe4,
       0x4a, 0xc8, 0x5a, 0x47, 0xf4, 0x7a, 0x8a, 0x1a, 0x34, 0xe3, 0xf0, 0x23,
       0xed, 0x6f, 0x73, 0x24, 0x3f, 0x01, 0x4d, 0x25, 0x7c, 0x7d, 0x06, 0x9e,
       0xe9, 0xdc, 0x16, 0xa9, 0x06, 0x25, 0xdc, 0x31, 0x40, 0xf7, 0x9f, 0x03,
       0xd4, 0xe1, 0x22, 0x77, 0x1b, 0x84, 0xb2, 0x02, 0x18, 0xc1, 0x54, 0xf6,
       0x28, 0xec, 0x81, 0x1d, 0x15, 0xbc, 0xa6, 0x8c, 0x9d, 0x67});
  auto rsa = crypto::keypair::PrivateKey::FromPrivateKeyInfo(kTestRSA1024Key);
  ASSERT_TRUE(rsa);
  // RSA-1024 is too small to be used with Android's format.
  EXPECT_EQ(AndroidRSAPublicKey(*rsa), std::nullopt);
}

TEST(AndroidRSATest, EncodePublicKeyTooLarge) {
  auto key = crypto::test::FixedRsa4096PrivateKeyForTesting();
  // RSA-4096 is too large to be used with Android's format.
  EXPECT_EQ(AndroidRSAPublicKey(key), std::nullopt);
}

TEST(AndroidRSATest, ValidPrehashedSignature) {
  auto privkey = crypto::test::FixedRsa2048PrivateKeyForTesting();
  auto pubkey = crypto::test::FixedRsa2048PublicKeyForTesting();

  constexpr auto kTestInput = std::to_array<uint8_t>({
      0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0xff, 0xfe, 0xfd,
      0xfc, 0xfb, 0xfa, 0xf9, 0xf8, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
      0x0e, 0x0f, 0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0,
  });

  // AndroidRSASign is supposed to compute a prehashed RSASSA-PKCS1-v1_5-SHA1
  // signature, so do that here, then check that the signature validates.
  const auto hash = crypto::hash::Sha1(kTestInput);
  std::string sig =
      AndroidRSASign(privkey, std::string(base::as_string_view(hash)));

  EXPECT_TRUE(crypto::sign::Verify(crypto::sign::SignatureKind::RSA_PKCS1_SHA1,
                                   pubkey, kTestInput,
                                   base::as_byte_span(sig)));
}

}  // namespace
