use std::collections::HashMap;

use async_lsp::lsp_types::{Diagnostic, DiagnosticSeverity};
use yara_x_parser::ast::dfs::{DFSEvent, DFSIter};
use yara_x_parser::ast::{Error, Expr, OfItems, PatternSet, WithSpan, AST};
use yara_x_parser::Span;

use crate::utils::position::span_to_range;

macro_rules! push_diagnostic {
    ($vec:expr, $span:expr, $message:expr, $text:expr, $severity:expr) => {
        $vec.push(Diagnostic {
            range: span_to_range($span, $text),
            message: $message,
            severity: Some($severity),
            ..Default::default()
        });
    };
}

/// Collects errors generated by [`yara_x_parser::ast::AST`].
fn collect_ast_errors(
    diagnostics: &mut Vec<Diagnostic>,
    ast: AST,
    text: &str,
) {
    let errors = ast.into_errors();
    for error in errors {
        match error {
            Error::SyntaxError { message, span }
            | Error::InvalidInteger { message, span }
            | Error::InvalidFloat { message, span }
            | Error::InvalidEscapeSequence { message, span }
            | Error::InvalidRegexpModifier { message, span } => {
                push_diagnostic!(
                    diagnostics,
                    span,
                    message,
                    text,
                    DiagnosticSeverity::ERROR
                );
            }
            Error::InvalidUTF8(span) => {
                let message = "Invalid UTF8".to_string();
                push_diagnostic!(
                    diagnostics,
                    span,
                    message,
                    text,
                    DiagnosticSeverity::ERROR
                );
            }
            Error::UnexpectedEscapeSequence(span) => {
                let message = "Unexpected escape sequence".to_string();
                push_diagnostic!(
                    diagnostics,
                    span,
                    message,
                    text,
                    DiagnosticSeverity::ERROR
                );
            }
        }
    }
}

/// Collects errors related to the use of patterns.
///
/// Finds undefined patterns and patterns that are defined but not used.
fn collect_variable_diagnostics(
    diagnostics: &mut Vec<Diagnostic>,
    ast: &AST,
    text: &str,
) {
    let rules = ast.rules();
    for rule in rules {
        let mut vars: HashMap<String, (bool, Span)> = HashMap::new();

        if let Some(patterns) = &rule.patterns {
            for pattern in patterns {
                if vars.contains_key(pattern.identifier().name) {
                    push_diagnostic!(
                        diagnostics,
                        pattern.identifier().span(),
                        "Pattern with this identifier is already defined"
                            .to_string(),
                        text,
                        DiagnosticSeverity::ERROR
                    );
                    continue;
                }

                vars.insert(
                    pattern.identifier().name.to_string(),
                    (false, pattern.identifier().span()),
                );
            }
        }

        let dfs = DFSIter::new(&rule.condition);

        for event in dfs {
            match event {
                DFSEvent::Enter(
                    expr @ (Expr::PatternMatch(_)
                    | Expr::PatternCount(_)
                    | Expr::PatternOffset(_)
                    | Expr::PatternLength(_)),
                ) => {
                    let name = match expr {
                        Expr::PatternMatch(pm) => {
                            pm.identifier.name.to_string()
                        }
                        Expr::PatternCount(ident_range) => {
                            format!("${}", &ident_range.identifier.name[1..])
                        }
                        Expr::PatternOffset(ident_index)
                        | Expr::PatternLength(ident_index) => {
                            format!("${}", &ident_index.identifier.name[1..])
                        }
                        _ => Default::default(),
                    };

                    if !vars.contains_key(&name) {
                        push_diagnostic!(
                            diagnostics,
                            expr.span(),
                            "Undefined pattern".to_string(),
                            text,
                            DiagnosticSeverity::ERROR
                        );
                        continue;
                    }
                    vars.entry(name)
                        .and_modify(|(is_used, _)| *is_used = true);
                }
                DFSEvent::Enter(expr @ (Expr::Of(_) | Expr::ForOf(_))) => {
                    let set = match expr {
                        Expr::Of(of_expr) => {
                            if let OfItems::PatternSet(PatternSet::Set(set)) =
                                &of_expr.items
                            {
                                Some(set)
                            } else {
                                None
                            }
                        }
                        Expr::ForOf(forof_expr) => {
                            if let PatternSet::Set(set) =
                                &forof_expr.pattern_set
                            {
                                Some(set)
                            } else {
                                None
                            }
                        }
                        _ => None,
                    };

                    if let Some(set) = set {
                        for item in set {
                            if item.wildcard {
                                vars.iter_mut().for_each(
                                    |(key, (is_used, _))| {
                                        if key.starts_with(item.identifier) {
                                            *is_used = true;
                                        }
                                    },
                                );
                            } else {
                                vars.entry(item.identifier.to_string())
                                    .and_modify(|(is_used, _)| {
                                        *is_used = true
                                    });
                            }
                        }
                    }
                }
                _ => {}
            }
        }

        for (is_used, span) in vars.into_values() {
            if !is_used {
                push_diagnostic!(
                    diagnostics,
                    span,
                    "This pattern is not used in condition block".to_string(),
                    text,
                    DiagnosticSeverity::HINT
                );
            }
        }
    }
}

/// Return diagnostic vector for the given source code.
pub fn diagnostics(ast: AST, src: &str) -> Vec<Diagnostic> {
    let mut diagnostics: Vec<Diagnostic> = Vec::new();
    collect_variable_diagnostics(&mut diagnostics, &ast, src);
    collect_ast_errors(&mut diagnostics, ast, src);

    diagnostics
}
