/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "gtest/gtest.h"
#include "nsCOMPtr.h"
#include "nsIPrefService.h"
#include "nsIX509CertDB.h"
#include "nsServiceManagerUtils.h"
#include "TrustOverrideUtils.h"

// certspec (for pycert.py)
//
// issuer:ca
// subject:ca
// extension:basicConstraints:cA,
// extension:keyUsage:cRLSign,keyCertSign
// serialNumber:1

const uint8_t kOverrideCaDer[] = {
    0x30, 0x82, 0x02, 0xB2, 0x30, 0x82, 0x01, 0x9C, 0xA0, 0x03, 0x02, 0x01,
    0x02, 0x02, 0x01, 0x01, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
    0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0C, 0x02, 0x63, 0x61, 0x30, 0x22, 0x18, 0x0F,
    0x32, 0x30, 0x31, 0x35, 0x31, 0x31, 0x32, 0x38, 0x30, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x5A, 0x18, 0x0F, 0x32, 0x30, 0x31, 0x38, 0x30, 0x32, 0x30,
    0x35, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5A, 0x30, 0x0D, 0x31, 0x0B,
    0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x02, 0x63, 0x61, 0x30,
    0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
    0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30,
    0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xBA, 0x88, 0x51, 0xA8,
    0x44, 0x8E, 0x16, 0xD6, 0x41, 0xFD, 0x6E, 0xB6, 0x88, 0x06, 0x36, 0x10,
    0x3D, 0x3C, 0x13, 0xD9, 0xEA, 0xE4, 0x35, 0x4A, 0xB4, 0xEC, 0xF5, 0x68,
    0x57, 0x6C, 0x24, 0x7B, 0xC1, 0xC7, 0x25, 0xA8, 0xE0, 0xD8, 0x1F, 0xBD,
    0xB1, 0x9C, 0x06, 0x9B, 0x6E, 0x1A, 0x86, 0xF2, 0x6B, 0xE2, 0xAF, 0x5A,
    0x75, 0x6B, 0x6A, 0x64, 0x71, 0x08, 0x7A, 0xA5, 0x5A, 0xA7, 0x45, 0x87,
    0xF7, 0x1C, 0xD5, 0x24, 0x9C, 0x02, 0x7E, 0xCD, 0x43, 0xFC, 0x1E, 0x69,
    0xD0, 0x38, 0x20, 0x29, 0x93, 0xAB, 0x20, 0xC3, 0x49, 0xE4, 0xDB, 0xB9,
    0x4C, 0xC2, 0x6B, 0x6C, 0x0E, 0xED, 0x15, 0x82, 0x0F, 0xF1, 0x7E, 0xAD,
    0x69, 0x1A, 0xB1, 0xD3, 0x02, 0x3A, 0x8B, 0x2A, 0x41, 0xEE, 0xA7, 0x70,
    0xE0, 0x0F, 0x0D, 0x8D, 0xFD, 0x66, 0x0B, 0x2B, 0xB0, 0x24, 0x92, 0xA4,
    0x7D, 0xB9, 0x88, 0x61, 0x79, 0x90, 0xB1, 0x57, 0x90, 0x3D, 0xD2, 0x3B,
    0xC5, 0xE0, 0xB8, 0x48, 0x1F, 0xA8, 0x37, 0xD3, 0x88, 0x43, 0xEF, 0x27,
    0x16, 0xD8, 0x55, 0xB7, 0x66, 0x5A, 0xAA, 0x7E, 0x02, 0x90, 0x2F, 0x3A,
    0x7B, 0x10, 0x80, 0x06, 0x24, 0xCC, 0x1C, 0x6C, 0x97, 0xAD, 0x96, 0x61,
    0x5B, 0xB7, 0xE2, 0x96, 0x12, 0xC0, 0x75, 0x31, 0xA3, 0x0C, 0x91, 0xDD,
    0xB4, 0xCA, 0xF7, 0xFC, 0xAD, 0x1D, 0x25, 0xD3, 0x09, 0xEF, 0xB9, 0x17,
    0x0E, 0xA7, 0x68, 0xE1, 0xB3, 0x7B, 0x2F, 0x22, 0x6F, 0x69, 0xE3, 0xB4,
    0x8A, 0x95, 0x61, 0x1D, 0xEE, 0x26, 0xD6, 0x25, 0x9D, 0xAB, 0x91, 0x08,
    0x4E, 0x36, 0xCB, 0x1C, 0x24, 0x04, 0x2C, 0xBF, 0x16, 0x8B, 0x2F, 0xE5,
    0xF1, 0x8F, 0x99, 0x17, 0x31, 0xB8, 0xB3, 0xFE, 0x49, 0x23, 0xFA, 0x72,
    0x51, 0xC4, 0x31, 0xD5, 0x03, 0xAC, 0xDA, 0x18, 0x0A, 0x35, 0xED, 0x8D,
    0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x1D, 0x30, 0x1B, 0x30, 0x0C, 0x06,
    0x03, 0x55, 0x1D, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30,
    0x0B, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06,
    0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
    0x0B, 0x03, 0x82, 0x01, 0x01, 0x00, 0x72, 0x11, 0xDF, 0xD7, 0x25, 0x7E,
    0x6E, 0x51, 0x3A, 0x7E, 0xF4, 0xB9, 0xDC, 0xBD, 0xC3, 0x1A, 0x6B, 0xF1,
    0x12, 0x99, 0xFD, 0xFA, 0xDF, 0xB1, 0xC0, 0xEB, 0xD6, 0x73, 0x30, 0x66,
    0x2C, 0x51, 0x33, 0x50, 0xF7, 0x67, 0x71, 0xA9, 0x1D, 0x69, 0x69, 0x50,
    0xBF, 0xBA, 0xBC, 0xD4, 0x13, 0xF2, 0x8C, 0x1A, 0xA4, 0x61, 0xB4, 0x22,
    0xD7, 0x95, 0xAC, 0xF3, 0x58, 0x58, 0xDD, 0xD9, 0xD7, 0x9F, 0xC2, 0xD3,
    0xD5, 0x09, 0x2D, 0x08, 0xBF, 0x60, 0xBA, 0xF7, 0x37, 0xCF, 0x87, 0xCC,
    0x75, 0x06, 0xD1, 0x8E, 0x98, 0xDC, 0x21, 0x17, 0x82, 0x62, 0xDD, 0xC4,
    0x3C, 0x1D, 0xB6, 0x26, 0x5D, 0x85, 0x0A, 0x6E, 0xA2, 0x8B, 0x85, 0xCB,
    0xD2, 0xA0, 0xC5, 0x0F, 0x6E, 0x96, 0x2B, 0x7C, 0xCC, 0x7B, 0x65, 0x24,
    0xB3, 0xAB, 0x79, 0xE5, 0xFF, 0x15, 0x78, 0x34, 0xC1, 0x18, 0xDA, 0x0F,
    0xA7, 0x62, 0x4C, 0xB1, 0x66, 0x35, 0x3E, 0x2E, 0x2F, 0x47, 0xAC, 0x0A,
    0x01, 0x8F, 0x7A, 0x5A, 0x85, 0xD2, 0xFE, 0xAA, 0xD8, 0x92, 0x63, 0x04,
    0x3A, 0x01, 0x6E, 0xF7, 0x4E, 0xE9, 0x02, 0x97, 0x80, 0x78, 0x4F, 0x15,
    0x27, 0x35, 0xE3, 0xA8, 0xC2, 0xC5, 0xEB, 0xF5, 0xB5, 0x54, 0xDE, 0xC8,
    0x4A, 0x71, 0xB3, 0x59, 0x0F, 0x33, 0x29, 0x09, 0xE5, 0xA7, 0x76, 0x00,
    0x9A, 0xF5, 0x99, 0x41, 0xC0, 0xFA, 0x20, 0xD2, 0xBC, 0x9A, 0xFA, 0x59,
    0xC4, 0x87, 0xAE, 0xE8, 0xDB, 0x47, 0x68, 0xA1, 0x17, 0x1B, 0x0E, 0xED,
    0x94, 0x67, 0xA4, 0x9A, 0x45, 0x61, 0x6A, 0x93, 0xED, 0x41, 0x96, 0x27,
    0x59, 0xCB, 0x17, 0xC9, 0xB4, 0x6B, 0x65, 0xFE, 0x60, 0xDE, 0x13, 0xA1,
    0x24, 0x53, 0xE5, 0xF2, 0x45, 0xB8, 0xD9, 0x5A, 0x31, 0x5C, 0xBD, 0x75,
    0x3C, 0xFE, 0x4D, 0x5F, 0x6C, 0xA5, 0x4E, 0xC2, 0x7F, 0xFE};

// certspec (for pycert.py)
//
// issuer:ca
// subject:ca-intermediate
// extension:basicConstraints:cA,
// extension:keyUsage:cRLSign,keyCertSign
// subjectKey:secp384r1
// serialNumber:2

const uint8_t kOverrideCaIntermediateDer[] = {
    0x30, 0x82, 0x02, 0x14, 0x30, 0x81, 0xFD, 0xA0, 0x03, 0x02, 0x01, 0x02,
    0x02, 0x01, 0x02, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
    0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x02, 0x63, 0x61, 0x30, 0x22, 0x18,
    0x0F, 0x32, 0x30, 0x31, 0x36, 0x31, 0x31, 0x32, 0x37, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x5A, 0x18, 0x0F, 0x32, 0x30, 0x31, 0x39, 0x30, 0x32,
    0x30, 0x35, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5A, 0x30, 0x1A, 0x31,
    0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x63, 0x61,
    0x2D, 0x69, 0x6E, 0x74, 0x65, 0x72, 0x6D, 0x65, 0x64, 0x69, 0x61, 0x74,
    0x65, 0x30, 0x76, 0x30, 0x10, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
    0x02, 0x01, 0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x22, 0x03, 0x62, 0x00,
    0x04, 0xA1, 0x68, 0x72, 0x43, 0x36, 0x2B, 0x5C, 0x7B, 0x18, 0x89, 0xF3,
    0x79, 0x15, 0x46, 0x15, 0xA1, 0xC7, 0x3F, 0xB4, 0x8D, 0xEE, 0x86, 0x3E,
    0x02, 0x29, 0x15, 0xDB, 0x60, 0x8E, 0x25, 0x2D, 0xE4, 0xB7, 0x13, 0x2D,
    0xA8, 0xCE, 0x98, 0xE8, 0x31, 0x53, 0x4E, 0x6A, 0x9C, 0x0C, 0x0B, 0x09,
    0xC8, 0xD6, 0x39, 0xAD, 0xE8, 0x32, 0x06, 0xE5, 0xBA, 0x81, 0x34, 0x73,
    0xA1, 0x1F, 0xA3, 0x30, 0xE0, 0x5D, 0xA8, 0xC9, 0x6E, 0x43, 0x83, 0xFE,
    0x27, 0x87, 0x3D, 0xA9, 0x71, 0x03, 0xBE, 0x28, 0x88, 0xCF, 0xF0, 0x02,
    0xF0, 0x5A, 0xF7, 0x1A, 0x1F, 0xDD, 0xCC, 0x83, 0x74, 0xAA, 0x6E, 0xA9,
    0xCE, 0xA3, 0x1D, 0x30, 0x1B, 0x30, 0x0C, 0x06, 0x03, 0x55, 0x1D, 0x13,
    0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30, 0x0B, 0x06, 0x03, 0x55,
    0x1D, 0x0F, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x0D, 0x06, 0x09,
    0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x03,
    0x82, 0x01, 0x01, 0x00, 0x0B, 0xC3, 0x68, 0x28, 0x89, 0x74, 0x5B, 0xEB,
    0xC2, 0x52, 0xC2, 0xA9, 0x33, 0x3B, 0xA8, 0x1F, 0x6C, 0x4B, 0xDB, 0xE3,
    0xCF, 0x79, 0x4B, 0xF3, 0x36, 0xD2, 0xB2, 0xC5, 0x25, 0xC7, 0x97, 0xD7,
    0xB4, 0x84, 0x00, 0x0A, 0x0D, 0x33, 0x25, 0x32, 0x94, 0xC8, 0xB2, 0x5F,
    0xFE, 0x74, 0xE5, 0x8A, 0xDF, 0xED, 0xC6, 0x13, 0xCE, 0xD1, 0x0E, 0xD3,
    0x3E, 0x32, 0xA9, 0x2A, 0x31, 0x38, 0xA2, 0x16, 0x0C, 0x10, 0x15, 0xBE,
    0x10, 0x7B, 0x3D, 0x53, 0x4F, 0xF2, 0x62, 0x8C, 0x55, 0xC2, 0x18, 0xDC,
    0x54, 0x65, 0x73, 0x11, 0x5B, 0x15, 0x41, 0x7C, 0xE0, 0x96, 0x5D, 0xF8,
    0xB7, 0x55, 0xB9, 0xAD, 0x5B, 0x71, 0xA1, 0xFE, 0x83, 0xAC, 0x7F, 0xE5,
    0x31, 0xCF, 0x58, 0xEE, 0xF8, 0x57, 0x15, 0x69, 0xD3, 0xF2, 0x3B, 0x22,
    0xA9, 0x7B, 0x5D, 0x21, 0x78, 0x77, 0x7E, 0x2F, 0x00, 0x8C, 0x2C, 0xD8,
    0x48, 0x08, 0xEC, 0x7D, 0x07, 0xF0, 0xA4, 0x74, 0x67, 0x33, 0xCB, 0x98,
    0xD8, 0x66, 0x80, 0x6F, 0xC4, 0xA8, 0xBD, 0x88, 0x86, 0x22, 0x4D, 0xAD,
    0x09, 0x27, 0x42, 0x06, 0x40, 0x3A, 0x57, 0xB8, 0x53, 0x79, 0x04, 0xB2,
    0x2C, 0x28, 0xF8, 0x26, 0xC0, 0xB3, 0x5F, 0x52, 0x73, 0x33, 0x1D, 0xF7,
    0x79, 0xBC, 0x37, 0x67, 0x5D, 0x0C, 0xAB, 0x60, 0x20, 0x46, 0x4D, 0xD4,
    0xC3, 0x9F, 0xCB, 0xF9, 0xBF, 0x8F, 0x29, 0x27, 0x19, 0x5F, 0x0E, 0x36,
    0x94, 0x93, 0x67, 0x89, 0x26, 0x5B, 0x1F, 0xB9, 0x47, 0xAF, 0x4F, 0x1B,
    0x5A, 0x15, 0x85, 0x5D, 0x52, 0xBC, 0x68, 0x3D, 0x95, 0x19, 0x47, 0x93,
    0xFC, 0x5B, 0x30, 0x4E, 0x5C, 0x77, 0x5C, 0x66, 0x15, 0x16, 0xB6, 0xD7,
    0x43, 0xC5, 0x83, 0x11, 0xED, 0x15, 0x90, 0x0F, 0xB8, 0x68, 0xDD, 0xF7,
    0xFA, 0x66, 0xD7, 0x3C, 0x19, 0xD6, 0xF1, 0xAA};

// /CN=ca
// SHA256 Fingerprint: A3:05:0C:44:CD:6D:1E:BE:A2:18:80:09:93:69:90:7F
//                     8C:E3:9F:A4:33:CB:E3:E9:3C:D1:8E:8C:89:23:1B:4A

// clang-format off
// Invocation: security/manager/tools/crtshToIdentifyingStruct/crtshToIdentifyingStruct.py -listname OverrideCaDNs -dn /tmp/overrideCa.pem
static const uint8_t CAcaDN[15] = {
  0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x02,
  0x63, 0x61,
};
// clang-format on

static const DataAndLength OverrideCaDNs[] = {
    {CAcaDN, sizeof(CAcaDN)},
};

// clang-format off
// Invocation: security/manager/tools/crtshToIdentifyingStruct/crtshToIdentifyingStruct.py -listname OverrideCaSPKIs -spki /tmp/overrideCa.pem
static const uint8_t CAcaSPKI[294] = {
  0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
  0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82,
  0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xBA, 0x88, 0x51, 0xA8, 0x44, 0x8E,
  0x16, 0xD6, 0x41, 0xFD, 0x6E, 0xB6, 0x88, 0x06, 0x36, 0x10, 0x3D, 0x3C, 0x13,
  0xD9, 0xEA, 0xE4, 0x35, 0x4A, 0xB4, 0xEC, 0xF5, 0x68, 0x57, 0x6C, 0x24, 0x7B,
  0xC1, 0xC7, 0x25, 0xA8, 0xE0, 0xD8, 0x1F, 0xBD, 0xB1, 0x9C, 0x06, 0x9B, 0x6E,
  0x1A, 0x86, 0xF2, 0x6B, 0xE2, 0xAF, 0x5A, 0x75, 0x6B, 0x6A, 0x64, 0x71, 0x08,
  0x7A, 0xA5, 0x5A, 0xA7, 0x45, 0x87, 0xF7, 0x1C, 0xD5, 0x24, 0x9C, 0x02, 0x7E,
  0xCD, 0x43, 0xFC, 0x1E, 0x69, 0xD0, 0x38, 0x20, 0x29, 0x93, 0xAB, 0x20, 0xC3,
  0x49, 0xE4, 0xDB, 0xB9, 0x4C, 0xC2, 0x6B, 0x6C, 0x0E, 0xED, 0x15, 0x82, 0x0F,
  0xF1, 0x7E, 0xAD, 0x69, 0x1A, 0xB1, 0xD3, 0x02, 0x3A, 0x8B, 0x2A, 0x41, 0xEE,
  0xA7, 0x70, 0xE0, 0x0F, 0x0D, 0x8D, 0xFD, 0x66, 0x0B, 0x2B, 0xB0, 0x24, 0x92,
  0xA4, 0x7D, 0xB9, 0x88, 0x61, 0x79, 0x90, 0xB1, 0x57, 0x90, 0x3D, 0xD2, 0x3B,
  0xC5, 0xE0, 0xB8, 0x48, 0x1F, 0xA8, 0x37, 0xD3, 0x88, 0x43, 0xEF, 0x27, 0x16,
  0xD8, 0x55, 0xB7, 0x66, 0x5A, 0xAA, 0x7E, 0x02, 0x90, 0x2F, 0x3A, 0x7B, 0x10,
  0x80, 0x06, 0x24, 0xCC, 0x1C, 0x6C, 0x97, 0xAD, 0x96, 0x61, 0x5B, 0xB7, 0xE2,
  0x96, 0x12, 0xC0, 0x75, 0x31, 0xA3, 0x0C, 0x91, 0xDD, 0xB4, 0xCA, 0xF7, 0xFC,
  0xAD, 0x1D, 0x25, 0xD3, 0x09, 0xEF, 0xB9, 0x17, 0x0E, 0xA7, 0x68, 0xE1, 0xB3,
  0x7B, 0x2F, 0x22, 0x6F, 0x69, 0xE3, 0xB4, 0x8A, 0x95, 0x61, 0x1D, 0xEE, 0x26,
  0xD6, 0x25, 0x9D, 0xAB, 0x91, 0x08, 0x4E, 0x36, 0xCB, 0x1C, 0x24, 0x04, 0x2C,
  0xBF, 0x16, 0x8B, 0x2F, 0xE5, 0xF1, 0x8F, 0x99, 0x17, 0x31, 0xB8, 0xB3, 0xFE,
  0x49, 0x23, 0xFA, 0x72, 0x51, 0xC4, 0x31, 0xD5, 0x03, 0xAC, 0xDA, 0x18, 0x0A,
  0x35, 0xED, 0x8D, 0x02, 0x03, 0x01, 0x00, 0x01,
};
// clang-format on

static const DataAndLength OverrideCaSPKIs[] = {
    {CAcaSPKI, sizeof(CAcaSPKI)},
};

class psm_TrustOverrideTest : public ::testing::Test {
 protected:
  void SetUp() override {
    nsCOMPtr<nsIPrefBranch> prefs(do_GetService(NS_PREFSERVICE_CONTRACTID));
    ASSERT_TRUE(prefs != nullptr)
    << "couldn't get nsIPrefBranch";

    // When PSM initializes, it attempts to get some localized strings.
    // As a result, Android flips out if this isn't set.
    nsresult rv = prefs->SetBoolPref("intl.locale.matchOS", true);
    ASSERT_TRUE(NS_SUCCEEDED(rv))
    << "couldn't set pref 'intl.locale.matchOS'";

    nsCOMPtr<nsIX509CertDB> certdb(do_GetService(NS_X509CERTDB_CONTRACTID));
    ASSERT_TRUE(certdb != nullptr)
    << "couldn't get certdb";
  }
};

TEST_F(psm_TrustOverrideTest, CheckCertDNIsInList) {
  nsTArray<uint8_t> caArray(kOverrideCaDer, sizeof(kOverrideCaDer));
  nsTArray<uint8_t> intermediateArray(kOverrideCaIntermediateDer,
                                      sizeof(kOverrideCaIntermediateDer));

  EXPECT_TRUE(CertDNIsInList(caArray, OverrideCaDNs))
      << "CA should be in the DN list";
  EXPECT_FALSE(CertDNIsInList(intermediateArray, OverrideCaDNs))
      << "Int should not be in the DN list";
}

TEST_F(psm_TrustOverrideTest, CheckCertSPKIIsInList) {
  mozilla::pkix::Input caInput;
  mozilla::pkix::Result rv =
      caInput.Init(kOverrideCaDer, sizeof(kOverrideCaDer));
  ASSERT_TRUE(rv == Success);

  mozilla::pkix::Input intermediateInput;
  rv = intermediateInput.Init(kOverrideCaIntermediateDer,
                              sizeof(kOverrideCaIntermediateDer));
  ASSERT_TRUE(rv == Success);

  EXPECT_TRUE(CertSPKIIsInList(caInput, OverrideCaSPKIs))
      << "CA should be in the SPKI list";
  EXPECT_FALSE(CertSPKIIsInList(intermediateInput, OverrideCaSPKIs))
      << "Int should not be in the SPKI list";
}
