#include "crc32iscsi.h"

uint32_t crc32iscsi_bit(uint32_t crc, void const *mem, size_t len) {
    uint8_t const *data = mem;
    if (data == NULL)
        return 0;
    crc = ~crc;
    for (size_t i = 0; i < len; i++) {
        crc ^= data[i];
        for (unsigned k = 0; k < 8; k++) {
            crc = crc & 1 ? (crc >> 1) ^ 0x82f63b78 : crc >> 1;
        }
    }
    crc = ~crc;
    return crc;
}

uint32_t crc32iscsi_rem(uint32_t crc, unsigned val, unsigned bits) {
    crc = ~crc;
    val &= (1U << bits) - 1;
    crc ^= val;
    for (unsigned i = 0; i < bits; i++) {
        crc = crc & 1 ? (crc >> 1) ^ 0x82f63b78 : crc >> 1;
    }
    crc = ~crc;
    return crc;
}

#define table_byte table_word[0]

static uint32_t const table_word[][256] = {
   {0x527d5351, 0xa016d052, 0xb34623a6, 0x412da0a5, 0x95e7c44e, 0x678c474d,
    0x74dcb4b9, 0x86b737ba, 0xd8a40b9e, 0x2acf889d, 0x399f7b69, 0xcbf4f86a,
    0x1f3e9c81, 0xed551f82, 0xfe05ec76, 0x0c6e6f75, 0x4223943e, 0xb048173d,
    0xa318e4c9, 0x517367ca, 0x85b90321, 0x77d28022, 0x648273d6, 0x96e9f0d5,
    0xc8faccf1, 0x3a914ff2, 0x29c1bc06, 0xdbaa3f05, 0x0f605bee, 0xfd0bd8ed,
    0xee5b2b19, 0x1c30a81a, 0x72c0dd8f, 0x80ab5e8c, 0x93fbad78, 0x61902e7b,
    0xb55a4a90, 0x4731c993, 0x54613a67, 0xa60ab964, 0xf8198540, 0x0a720643,
    0x1922f5b7, 0xeb4976b4, 0x3f83125f, 0xcde8915c, 0xdeb862a8, 0x2cd3e1ab,
    0x629e1ae0, 0x90f599e3, 0x83a56a17, 0x71cee914, 0xa5048dff, 0x576f0efc,
    0x443ffd08, 0xb6547e0b, 0xe847422f, 0x1a2cc12c, 0x097c32d8, 0xfb17b1db,
    0x2fddd530, 0xddb65633, 0xcee6a5c7, 0x3c8d26c4, 0x13064eed, 0xe16dcdee,
    0xf23d3e1a, 0x0056bd19, 0xd49cd9f2, 0x26f75af1, 0x35a7a905, 0xc7cc2a06,
    0x99df1622, 0x6bb49521, 0x78e466d5, 0x8a8fe5d6, 0x5e45813d, 0xac2e023e,
    0xbf7ef1ca, 0x4d1572c9, 0x03588982, 0xf1330a81, 0xe263f975, 0x10087a76,
    0xc4c21e9d, 0x36a99d9e, 0x25f96e6a, 0xd792ed69, 0x8981d14d, 0x7bea524e,
    0x68baa1ba, 0x9ad122b9, 0x4e1b4652, 0xbc70c551, 0xaf2036a5, 0x5d4bb5a6,
    0x33bbc033, 0xc1d04330, 0xd280b0c4, 0x20eb33c7, 0xf421572c, 0x064ad42f,
    0x151a27db, 0xe771a4d8, 0xb96298fc, 0x4b091bff, 0x5859e80b, 0xaa326b08,
    0x7ef80fe3, 0x8c938ce0, 0x9fc37f14, 0x6da8fc17, 0x23e5075c, 0xd18e845f,
    0xc2de77ab, 0x30b5f4a8, 0xe47f9043, 0x16141340, 0x0544e0b4, 0xf72f63b7,
    0xa93c5f93, 0x5b57dc90, 0x48072f64, 0xba6cac67, 0x6ea6c88c, 0x9ccd4b8f,
    0x8f9db87b, 0x7df63b78, 0xd08b6829, 0x22e0eb2a, 0x31b018de, 0xc3db9bdd,
    0x1711ff36, 0xe57a7c35, 0xf62a8fc1, 0x04410cc2, 0x5a5230e6, 0xa839b3e5,
    0xbb694011, 0x4902c312, 0x9dc8a7f9, 0x6fa324fa, 0x7cf3d70e, 0x8e98540d,
    0xc0d5af46, 0x32be2c45, 0x21eedfb1, 0xd3855cb2, 0x074f3859, 0xf524bb5a,
    0xe67448ae, 0x141fcbad, 0x4a0cf789, 0xb867748a, 0xab37877e, 0x595c047d,
    0x8d966096, 0x7ffde395, 0x6cad1061, 0x9ec69362, 0xf036e6f7, 0x025d65f4,
    0x110d9600, 0xe3661503, 0x37ac71e8, 0xc5c7f2eb, 0xd697011f, 0x24fc821c,
    0x7aefbe38, 0x88843d3b, 0x9bd4cecf, 0x69bf4dcc, 0xbd752927, 0x4f1eaa24,
    0x5c4e59d0, 0xae25dad3, 0xe0682198, 0x1203a29b, 0x0153516f, 0xf338d26c,
    0x27f2b687, 0xd5993584, 0xc6c9c670, 0x34a24573, 0x6ab17957, 0x98dafa54,
    0x8b8a09a0, 0x79e18aa3, 0xad2bee48, 0x5f406d4b, 0x4c109ebf, 0xbe7b1dbc,
    0x91f07595, 0x639bf696, 0x70cb0562, 0x82a08661, 0x566ae28a, 0xa4016189,
    0xb751927d, 0x453a117e, 0x1b292d5a, 0xe942ae59, 0xfa125dad, 0x0879deae,
    0xdcb3ba45, 0x2ed83946, 0x3d88cab2, 0xcfe349b1, 0x81aeb2fa, 0x73c531f9,
    0x6095c20d, 0x92fe410e, 0x463425e5, 0xb45fa6e6, 0xa70f5512, 0x5564d611,
    0x0b77ea35, 0xf91c6936, 0xea4c9ac2, 0x182719c1, 0xcced7d2a, 0x3e86fe29,
    0x2dd60ddd, 0xdfbd8ede, 0xb14dfb4b, 0x43267848, 0x50768bbc, 0xa21d08bf,
    0x76d76c54, 0x84bcef57, 0x97ec1ca3, 0x65879fa0, 0x3b94a384, 0xc9ff2087,
    0xdaafd373, 0x28c45070, 0xfc0e349b, 0x0e65b798, 0x1d35446c, 0xef5ec76f,
    0xa1133c24, 0x5378bf27, 0x40284cd3, 0xb243cfd0, 0x6689ab3b, 0x94e22838,
    0x87b2dbcc, 0x75d958cf, 0x2bca64eb, 0xd9a1e7e8, 0xcaf1141c, 0x389a971f,
    0xec50f3f4, 0x1e3b70f7, 0x0d6b8303, 0xff000000},
   {0xa31c2483, 0xb0bebcf4, 0x8459146d, 0x97fb8c1a, 0xed96455f, 0xfe34dd28,
    0xcad375b1, 0xd971edc6, 0x3e08e73b, 0x2daa7f4c, 0x194dd7d5, 0x0aef4fa2,
    0x708286e7, 0x63201e90, 0x57c7b609, 0x44652e7e, 0x9cd9d502, 0x8f7b4d75,
    0xbb9ce5ec, 0xa83e7d9b, 0xd253b4de, 0xc1f12ca9, 0xf5168430, 0xe6b41c47,
    0x01cd16ba, 0x126f8ecd, 0x26882654, 0x352abe23, 0x4f477766, 0x5ce5ef11,
    0x68024788, 0x7ba0dfff, 0xdc97c781, 0xcf355ff6, 0xfbd2f76f, 0xe8706f18,
    0x921da65d, 0x81bf3e2a, 0xb55896b3, 0xa6fa0ec4, 0x41830439, 0x52219c4e,
    0x66c634d7, 0x7564aca0, 0x0f0965e5, 0x1cabfd92, 0x284c550b, 0x3beecd7c,
    0xe3523600, 0xf0f0ae77, 0xc41706ee, 0xd7b59e99, 0xadd857dc, 0xbe7acfab,
    0x8a9d6732, 0x993fff45, 0x7e46f5b8, 0x6de46dcf, 0x5903c556, 0x4aa15d21,
    0x30cc9464, 0x236e0c13, 0x1789a48a, 0x042b3cfd, 0x5c0be287, 0x4fa97af0,
    0x7b4ed269, 0x68ec4a1e, 0x1281835b, 0x01231b2c, 0x35c4b3b5, 0x26662bc2,
    0xc11f213f, 0xd2bdb948, 0xe65a11d1, 0xf5f889a6, 0x8f9540e3, 0x9c37d894,
    0xa8d0700d, 0xbb72e87a, 0x63ce1306, 0x706c8b71, 0x448b23e8, 0x5729bb9f,
    0x2d4472da, 0x3ee6eaad, 0x0a014234, 0x19a3da43, 0xfedad0be, 0xed7848c9,
    0xd99fe050, 0xca3d7827, 0xb050b162, 0xa3f22915, 0x9715818c, 0x84b719fb,
    0x23800185, 0x302299f2, 0x04c5316b, 0x1767a91c, 0x6d0a6059, 0x7ea8f82e,
    0x4a4f50b7, 0x59edc8c0, 0xbe94c23d, 0xad365a4a, 0x99d1f2d3, 0x8a736aa4,
    0xf01ea3e1, 0xe3bc3b96, 0xd75b930f, 0xc4f90b78, 0x1c45f004, 0x0fe76873,
    0x3b00c0ea, 0x28a2589d, 0x52cf91d8, 0x416d09af, 0x758aa136, 0x66283941,
    0x815133bc, 0x92f3abcb, 0xa6140352, 0xb5b69b25, 0xcfdb5260, 0xdc79ca17,
    0xe89e628e, 0xfb3cfaf9, 0x58dfde7a, 0x4b7d460d, 0x7f9aee94, 0x6c3876e3,
    0x1655bfa6, 0x05f727d1, 0x31108f48, 0x22b2173f, 0xc5cb1dc2, 0xd66985b5,
    0xe28e2d2c, 0xf12cb55b, 0x8b417c1e, 0x98e3e469, 0xac044cf0, 0xbfa6d487,
    0x671a2ffb, 0x74b8b78c, 0x405f1f15, 0x53fd8762, 0x29904e27, 0x3a32d650,
    0x0ed57ec9, 0x1d77e6be, 0xfa0eec43, 0xe9ac7434, 0xdd4bdcad, 0xcee944da,
    0xb4848d9f, 0xa72615e8, 0x93c1bd71, 0x80632506, 0x27543d78, 0x34f6a50f,
    0x00110d96, 0x13b395e1, 0x69de5ca4, 0x7a7cc4d3, 0x4e9b6c4a, 0x5d39f43d,
    0xba40fec0, 0xa9e266b7, 0x9d05ce2e, 0x8ea75659, 0xf4ca9f1c, 0xe768076b,
    0xd38faff2, 0xc02d3785, 0x1891ccf9, 0x0b33548e, 0x3fd4fc17, 0x2c766460,
    0x561bad25, 0x45b93552, 0x715e9dcb, 0x62fc05bc, 0x85850f41, 0x96279736,
    0xa2c03faf, 0xb162a7d8, 0xcb0f6e9d, 0xd8adf6ea, 0xec4a5e73, 0xffe8c604,
    0xa7c8187e, 0xb46a8009, 0x808d2890, 0x932fb0e7, 0xe94279a2, 0xfae0e1d5,
    0xce07494c, 0xdda5d13b, 0x3adcdbc6, 0x297e43b1, 0x1d99eb28, 0x0e3b735f,
    0x7456ba1a, 0x67f4226d, 0x53138af4, 0x40b11283, 0x980de9ff, 0x8baf7188,
    0xbf48d911, 0xacea4166, 0xd6878823, 0xc5251054, 0xf1c2b8cd, 0xe26020ba,
    0x05192a47, 0x16bbb230, 0x225c1aa9, 0x31fe82de, 0x4b934b9b, 0x5831d3ec,
    0x6cd67b75, 0x7f74e302, 0xd843fb7c, 0xcbe1630b, 0xff06cb92, 0xeca453e5,
    0x96c99aa0, 0x856b02d7, 0xb18caa4e, 0xa22e3239, 0x455738c4, 0x56f5a0b3,
    0x6212082a, 0x71b0905d, 0x0bdd5918, 0x187fc16f, 0x2c9869f6, 0x3f3af181,
    0xe7860afd, 0xf424928a, 0xc0c33a13, 0xd361a264, 0xa90c6b21, 0xbaaef356,
    0x8e495bcf, 0x9debc3b8, 0x7a92c945, 0x69305132, 0x5dd7f9ab, 0x4e7561dc,
    0x3418a899, 0x27ba30ee, 0x135d9877, 0x00ff0000},
   {0x9105d4a8, 0x344446d6, 0xde6a86a5, 0x7b2b14db, 0x0fdb70b2, 0xaa9ae2cc,
    0x40b422bf, 0xe5f5b0c1, 0xa954ea6d, 0x0c157813, 0xe63bb860, 0x437a2a1e,
    0x378a4e77, 0x92cbdc09, 0x78e51c7a, 0xdda48e04, 0xe1a7a922, 0x44e63b5c,
    0xaec8fb2f, 0x0b896951, 0x7f790d38, 0xda389f46, 0x30165f35, 0x9557cd4b,
    0xd9f697e7, 0x7cb70599, 0x9699c5ea, 0x33d85794, 0x472833fd, 0xe269a183,
    0x084761f0, 0xad06f38e, 0x70412fbc, 0xd500bdc2, 0x3f2e7db1, 0x9a6fefcf,
    0xee9f8ba6, 0x4bde19d8, 0xa1f0d9ab, 0x04b14bd5, 0x48101179, 0xed518307,
    0x077f4374, 0xa23ed10a, 0xd6ceb563, 0x738f271d, 0x99a1e76e, 0x3ce07510,
    0x00e35236, 0xa5a2c048, 0x4f8c003b, 0xeacd9245, 0x9e3df62c, 0x3b7c6452,
    0xd152a421, 0x7413365f, 0x38b26cf3, 0x9df3fe8d, 0x77dd3efe, 0xd29cac80,
    0xa66cc8e9, 0x032d5a97, 0xe9039ae4, 0x4c42089a, 0x56605471, 0xf321c60f,
    0x190f067c, 0xbc4e9402, 0xc8bef06b, 0x6dff6215, 0x87d1a266, 0x22903018,
    0x6e316ab4, 0xcb70f8ca, 0x215e38b9, 0x841faac7, 0xf0efceae, 0x55ae5cd0,
    0xbf809ca3, 0x1ac10edd, 0x26c229fb, 0x8383bb85, 0x69ad7bf6, 0xccece988,
    0xb81c8de1, 0x1d5d1f9f, 0xf773dfec, 0x52324d92, 0x1e93173e, 0xbbd28540,
    0x51fc4533, 0xf4bdd74d, 0x804db324, 0x250c215a, 0xcf22e129, 0x6a637357,
    0xb724af65, 0x12653d1b, 0xf84bfd68, 0x5d0a6f16, 0x29fa0b7f, 0x8cbb9901,
    0x66955972, 0xc3d4cb0c, 0x8f7591a0, 0x2a3403de, 0xc01ac3ad, 0x655b51d3,
    0x11ab35ba, 0xb4eaa7c4, 0x5ec467b7, 0xfb85f5c9, 0xc786d2ef, 0x62c74091,
    0x88e980e2, 0x2da8129c, 0x595876f5, 0xfc19e48b, 0x163724f8, 0xb376b686,
    0xffd7ec2a, 0x5a967e54, 0xb0b8be27, 0x15f92c59, 0x61094830, 0xc448da4e,
    0x2e661a3d, 0x8b278843, 0x1a22a3eb, 0xbf633195, 0x554df1e6, 0xf00c6398,
    0x84fc07f1, 0x21bd958f, 0xcb9355fc, 0x6ed2c782, 0x22739d2e, 0x87320f50,
    0x6d1ccf23, 0xc85d5d5d, 0xbcad3934, 0x19ecab4a, 0xf3c26b39, 0x5683f947,
    0x6a80de61, 0xcfc14c1f, 0x25ef8c6c, 0x80ae1e12, 0xf45e7a7b, 0x511fe805,
    0xbb312876, 0x1e70ba08, 0x52d1e0a4, 0xf79072da, 0x1dbeb2a9, 0xb8ff20d7,
    0xcc0f44be, 0x694ed6c0, 0x836016b3, 0x262184cd, 0xfb6658ff, 0x5e27ca81,
    0xb4090af2, 0x1148988c, 0x65b8fce5, 0xc0f96e9b, 0x2ad7aee8, 0x8f963c96,
    0xc337663a, 0x6676f444, 0x8c583437, 0x2919a649, 0x5de9c220, 0xf8a8505e,
    0x1286902d, 0xb7c70253, 0x8bc42575, 0x2e85b70b, 0xc4ab7778, 0x61eae506,
    0x151a816f, 0xb05b1311, 0x5a75d362, 0xff34411c, 0xb3951bb0, 0x16d489ce,
    0xfcfa49bd, 0x59bbdbc3, 0x2d4bbfaa, 0x880a2dd4, 0x6224eda7, 0xc7657fd9,
    0xdd472332, 0x7806b14c, 0x9228713f, 0x3769e341, 0x43998728, 0xe6d81556,
    0x0cf6d525, 0xa9b7475b, 0xe5161df7, 0x40578f89, 0xaa794ffa, 0x0f38dd84,
    0x7bc8b9ed, 0xde892b93, 0x34a7ebe0, 0x91e6799e, 0xade55eb8, 0x08a4ccc6,
    0xe28a0cb5, 0x47cb9ecb, 0x333bfaa2, 0x967a68dc, 0x7c54a8af, 0xd9153ad1,
    0x95b4607d, 0x30f5f203, 0xdadb3270, 0x7f9aa00e, 0x0b6ac467, 0xae2b5619,
    0x4405966a, 0xe1440414, 0x3c03d826, 0x99424a58, 0x736c8a2b, 0xd62d1855,
    0xa2dd7c3c, 0x079cee42, 0xedb22e31, 0x48f3bc4f, 0x0452e6e3, 0xa113749d,
    0x4b3db4ee, 0xee7c2690, 0x9a8c42f9, 0x3fcdd087, 0xd5e310f4, 0x70a2828a,
    0x4ca1a5ac, 0xe9e037d2, 0x03cef7a1, 0xa68f65df, 0xd27f01b6, 0x773e93c8,
    0x9d1053bb, 0x3851c1c5, 0x74f09b69, 0xd1b10917, 0x3b9fc964, 0x9ede5b1a,
    0xea2e3f73, 0x4f6fad0d, 0xa5416d7e, 0x0000ff00},
   {0x2803e8bd, 0xf5464205, 0x9764cb3c, 0x4a216184, 0x5321d94e, 0x8e6473f6,
    0xec46facf, 0x31035077, 0xde478b5b, 0x030221e3, 0x6120a8da, 0xbc650262,
    0xa565baa8, 0x78201010, 0x1a029929, 0xc7473391, 0xc1675980, 0x1c22f338,
    0x7e007a01, 0xa345d0b9, 0xba456873, 0x6700c2cb, 0x05224bf2, 0xd867e14a,
    0x37233a66, 0xea6690de, 0x884419e7, 0x5501b35f, 0x4c010b95, 0x9144a12d,
    0xf3662814, 0x2e2382ac, 0xff26fc36, 0x2263568e, 0x4041dfb7, 0x9d04750f,
    0x8404cdc5, 0x5941677d, 0x3b63ee44, 0xe62644fc, 0x09629fd0, 0xd4273568,
    0xb605bc51, 0x6b4016e9, 0x7240ae23, 0xaf05049b, 0xcd278da2, 0x1062271a,
    0x16424d0b, 0xcb07e7b3, 0xa9256e8a, 0x7460c432, 0x6d607cf8, 0xb025d640,
    0xd2075f79, 0x0f42f5c1, 0xe0062eed, 0x3d438455, 0x5f610d6c, 0x8224a7d4,
    0x9b241f1e, 0x4661b5a6, 0x24433c9f, 0xf9069627, 0x83a5b75a, 0x5ee01de2,
    0x3cc294db, 0xe1873e63, 0xf88786a9, 0x25c22c11, 0x47e0a528, 0x9aa50f90,
    0x75e1d4bc, 0xa8a47e04, 0xca86f73d, 0x17c35d85, 0x0ec3e54f, 0xd3864ff7,
    0xb1a4c6ce, 0x6ce16c76, 0x6ac10667, 0xb784acdf, 0xd5a625e6, 0x08e38f5e,
    0x11e33794, 0xcca69d2c, 0xae841415, 0x73c1bead, 0x9c856581, 0x41c0cf39,
    0x23e24600, 0xfea7ecb8, 0xe7a75472, 0x3ae2feca, 0x58c077f3, 0x8585dd4b,
    0x5480a3d1, 0x89c50969, 0xebe78050, 0x36a22ae8, 0x2fa29222, 0xf2e7389a,
    0x90c5b1a3, 0x4d801b1b, 0xa2c4c037, 0x7f816a8f, 0x1da3e3b6, 0xc0e6490e,
    0xd9e6f1c4, 0x04a35b7c, 0x6681d245, 0xbbc478fd, 0xbde412ec, 0x60a1b854,
    0x0283316d, 0xdfc69bd5, 0xc6c6231f, 0x1b8389a7, 0x79a1009e, 0xa4e4aa26,
    0x4ba0710a, 0x96e5dbb2, 0xf4c7528b, 0x2982f833, 0x308240f9, 0xedc7ea41,
    0x8fe56378, 0x52a0c9c0, 0x7aa32182, 0xa7e68b3a, 0xc5c40203, 0x1881a8bb,
    0x01811071, 0xdcc4bac9, 0xbee633f0, 0x63a39948, 0x8ce74264, 0x51a2e8dc,
    0x338061e5, 0xeec5cb5d, 0xf7c57397, 0x2a80d92f, 0x48a25016, 0x95e7faae,
    0x93c790bf, 0x4e823a07, 0x2ca0b33e, 0xf1e51986, 0xe8e5a14c, 0x35a00bf4,
    0x578282cd, 0x8ac72875, 0x6583f359, 0xb8c659e1, 0xdae4d0d8, 0x07a17a60,
    0x1ea1c2aa, 0xc3e46812, 0xa1c6e12b, 0x7c834b93, 0xad863509, 0x70c39fb1,
    0x12e11688, 0xcfa4bc30, 0xd6a404fa, 0x0be1ae42, 0x69c3277b, 0xb4868dc3,
    0x5bc256ef, 0x8687fc57, 0xe4a5756e, 0x39e0dfd6, 0x20e0671c, 0xfda5cda4,
    0x9f87449d, 0x42c2ee25, 0x44e28434, 0x99a72e8c, 0xfb85a7b5, 0x26c00d0d,
    0x3fc0b5c7, 0xe2851f7f, 0x80a79646, 0x5de23cfe, 0xb2a6e7d2, 0x6fe34d6a,
    0x0dc1c453, 0xd0846eeb, 0xc984d621, 0x14c17c99, 0x76e3f5a0, 0xaba65f18,
    0xd1057e65, 0x0c40d4dd, 0x6e625de4, 0xb327f75c, 0xaa274f96, 0x7762e52e,
    0x15406c17, 0xc805c6af, 0x27411d83, 0xfa04b73b, 0x98263e02, 0x456394ba,
    0x5c632c70, 0x812686c8, 0xe3040ff1, 0x3e41a549, 0x3861cf58, 0xe52465e0,
    0x8706ecd9, 0x5a434661, 0x4343feab, 0x9e065413, 0xfc24dd2a, 0x21617792,
    0xce25acbe, 0x13600606, 0x71428f3f, 0xac072587, 0xb5079d4d, 0x684237f5,
    0x0a60becc, 0xd7251474, 0x06206aee, 0xdb65c056, 0xb947496f, 0x6402e3d7,
    0x7d025b1d, 0xa047f1a5, 0xc265789c, 0x1f20d224, 0xf0640908, 0x2d21a3b0,
    0x4f032a89, 0x92468031, 0x8b4638fb, 0x56039243, 0x34211b7a, 0xe964b1c2,
    0xef44dbd3, 0x3201716b, 0x5023f852, 0x8d6652ea, 0x9466ea20, 0x49234098,
    0x2b01c9a1, 0xf6446319, 0x1900b835, 0xc445128d, 0xa6679bb4, 0x7b22310c,
    0x622289c6, 0xbf67237e, 0xdd45aa47, 0x000000ff},
   {0x0d153df2, 0x3504525e, 0x7d37e2aa, 0x45268d06, 0xed508342, 0xd541ecee,
    0x9d725c1a, 0xa56333b6, 0xc8723663, 0xf06359cf, 0xb850e93b, 0x80418697,
    0x283788d3, 0x1026e77f, 0x5815578b, 0x60043827, 0x82375c21, 0xba26338d,
    0xf2158379, 0xca04ecd5, 0x6272e291, 0x5a638d3d, 0x12503dc9, 0x2a415265,
    0x475057b0, 0x7f41381c, 0x377288e8, 0x0f63e744, 0xa715e900, 0x9f0486ac,
    0xd7373658, 0xef2659f4, 0x16bd88a5, 0x2eace709, 0x669f57fd, 0x5e8e3851,
    0xf6f83615, 0xcee959b9, 0x86dae94d, 0xbecb86e1, 0xd3da8334, 0xebcbec98,
    0xa3f85c6c, 0x9be933c0, 0x339f3d84, 0x0b8e5228, 0x43bde2dc, 0x7bac8d70,
    0x999fe976, 0xa18e86da, 0xe9bd362e, 0xd1ac5982, 0x79da57c6, 0x41cb386a,
    0x09f8889e, 0x31e9e732, 0x5cf8e2e7, 0x64e98d4b, 0x2cda3dbf, 0x14cb5213,
    0xbcbd5c57, 0x84ac33fb, 0xcc9f830f, 0xf48eeca3, 0x3a44575c, 0x025538f0,
    0x4a668804, 0x7277e7a8, 0xda01e9ec, 0xe2108640, 0xaa2336b4, 0x92325918,
    0xff235ccd, 0xc7323361, 0x8f018395, 0xb710ec39, 0x1f66e27d, 0x27778dd1,
    0x6f443d25, 0x57555289, 0xb566368f, 0x8d775923, 0xc544e9d7, 0xfd55867b,
    0x5523883f, 0x6d32e793, 0x25015767, 0x1d1038cb, 0x70013d1e, 0x481052b2,
    0x0023e246, 0x38328dea, 0x904483ae, 0xa855ec02, 0xe0665cf6, 0xd877335a,
    0x21ece20b, 0x19fd8da7, 0x51ce3d53, 0x69df52ff, 0xc1a95cbb, 0xf9b83317,
    0xb18b83e3, 0x899aec4f, 0xe48be99a, 0xdc9a8636, 0x94a936c2, 0xacb8596e,
    0x04ce572a, 0x3cdf3886, 0x74ec8872, 0x4cfde7de, 0xaece83d8, 0x96dfec74,
    0xdeec5c80, 0xe6fd332c, 0x4e8b3d68, 0x769a52c4, 0x3ea9e230, 0x06b88d9c,
    0x6ba98849, 0x53b8e7e5, 0x1b8b5711, 0x239a38bd, 0x8bec36f9, 0xb3fd5955,
    0xfbcee9a1, 0xc3df860d, 0x63b7e8ae, 0x5ba68702, 0x139537f6, 0x2b84585a,
    0x83f2561e, 0xbbe339b2, 0xf3d08946, 0xcbc1e6ea, 0xa6d0e33f, 0x9ec18c93,
    0xd6f23c67, 0xeee353cb, 0x46955d8f, 0x7e843223, 0x36b782d7, 0x0ea6ed7b,
    0xec95897d, 0xd484e6d1, 0x9cb75625, 0xa4a63989, 0x0cd037cd, 0x34c15861,
    0x7cf2e895, 0x44e38739, 0x29f282ec, 0x11e3ed40, 0x59d05db4, 0x61c13218,
    0xc9b73c5c, 0xf1a653f0, 0xb995e304, 0x81848ca8, 0x781f5df9, 0x400e3255,
    0x083d82a1, 0x302ced0d, 0x985ae349, 0xa04b8ce5, 0xe8783c11, 0xd06953bd,
    0xbd785668, 0x856939c4, 0xcd5a8930, 0xf54be69c, 0x5d3de8d8, 0x652c8774,
    0x2d1f3780, 0x150e582c, 0xf73d3c2a, 0xcf2c5386, 0x871fe372, 0xbf0e8cde,
    0x1778829a, 0x2f69ed36, 0x675a5dc2, 0x5f4b326e, 0x325a37bb, 0x0a4b5817,
    0x4278e8e3, 0x7a69874f, 0xd21f890b, 0xea0ee6a7, 0xa23d5653, 0x9a2c39ff,
    0x54e68200, 0x6cf7edac, 0x24c45d58, 0x1cd532f4, 0xb4a33cb0, 0x8cb2531c,
    0xc481e3e8, 0xfc908c44, 0x91818991, 0xa990e63d, 0xe1a356c9, 0xd9b23965,
    0x71c43721, 0x49d5588d, 0x01e6e879, 0x39f787d5, 0xdbc4e3d3, 0xe3d58c7f,
    0xabe63c8b, 0x93f75327, 0x3b815d63, 0x039032cf, 0x4ba3823b, 0x73b2ed97,
    0x1ea3e842, 0x26b287ee, 0x6e81371a, 0x569058b6, 0xfee656f2, 0xc6f7395e,
    0x8ec489aa, 0xb6d5e606, 0x4f4e3757, 0x775f58fb, 0x3f6ce80f, 0x077d87a3,
    0xaf0b89e7, 0x971ae64b, 0xdf2956bf, 0xe7383913, 0x8a293cc6, 0xb238536a,
    0xfa0be39e, 0xc21a8c32, 0x6a6c8276, 0x527dedda, 0x1a4e5d2e, 0x225f3282,
    0xc06c5684, 0xf87d3928, 0xb04e89dc, 0x885fe670, 0x2029e834, 0x18388798,
    0x500b376c, 0x681a58c0, 0x050b5d15, 0x3d1a32b9, 0x7529824d, 0x4d38ede1,
    0xe54ee3a5, 0xdd5f8c09, 0x956c3cfd, 0xad7d5351},
   {0x12580abf, 0xfd6861a6, 0xc9d4aa7c, 0x26e4c165, 0xa0ad3dc8, 0x4f9d56d1,
    0x7b219d0b, 0x9411f612, 0x725e12a0, 0x9d6e79b9, 0xa9d2b263, 0x46e2d97a,
    0xc0ab25d7, 0x2f9b4ece, 0x1b278514, 0xf417ee0d, 0xd2543a81, 0x3d645198,
    0x09d89a42, 0xe6e8f15b, 0x60a10df6, 0x8f9166ef, 0xbb2dad35, 0x541dc62c,
    0xb252229e, 0x5d624987, 0x69de825d, 0x86eee944, 0x00a715e9, 0xef977ef0,
    0xdb2bb52a, 0x341bde33, 0x97ac1c32, 0x789c772b, 0x4c20bcf1, 0xa310d7e8,
    0x25592b45, 0xca69405c, 0xfed58b86, 0x11e5e09f, 0xf7aa042d, 0x189a6f34,
    0x2c26a4ee, 0xc316cff7, 0x455f335a, 0xaa6f5843, 0x9ed39399, 0x71e3f880,
    0x57a02c0c, 0xb8904715, 0x8c2c8ccf, 0x631ce7d6, 0xe5551b7b, 0x0a657062,
    0x3ed9bbb8, 0xd1e9d0a1, 0x37a63413, 0xd8965f0a, 0xec2a94d0, 0x031affc9,
    0x85530364, 0x6a63687d, 0x5edfa3a7, 0xb1efc8be, 0x1c5c5154, 0xf36c3a4d,
    0xc7d0f197, 0x28e09a8e, 0xaea96623, 0x41990d3a, 0x7525c6e0, 0x9a15adf9,
    0x7c5a494b, 0x936a2252, 0xa7d6e988, 0x48e68291, 0xceaf7e3c, 0x219f1525,
    0x1523deff, 0xfa13b5e6, 0xdc50616a, 0x33600a73, 0x07dcc1a9, 0xe8ecaab0,
    0x6ea5561d, 0x81953d04, 0xb529f6de, 0x5a199dc7, 0xbc567975, 0x5366126c,
    0x67dad9b6, 0x88eab2af, 0x0ea34e02, 0xe193251b, 0xd52feec1, 0x3a1f85d8,
    0x99a847d9, 0x76982cc0, 0x4224e71a, 0xad148c03, 0x2b5d70ae, 0xc46d1bb7,
    0xf0d1d06d, 0x1fe1bb74, 0xf9ae5fc6, 0x169e34df, 0x2222ff05, 0xcd12941c,
    0x4b5b68b1, 0xa46b03a8, 0x90d7c872, 0x7fe7a36b, 0x59a477e7, 0xb6941cfe,
    0x8228d724, 0x6d18bc3d, 0xeb514090, 0x04612b89, 0x30dde053, 0xdfed8b4a,
    0x39a26ff8, 0xd69204e1, 0xe22ecf3b, 0x0d1ea422, 0x8b57588f, 0x64673396,
    0x50dbf84c, 0xbfeb9355, 0x0e50bd69, 0xe160d670, 0xd5dc1daa, 0x3aec76b3,
    0xbca58a1e, 0x5395e107, 0x67292add, 0x881941c4, 0x6e56a576, 0x8166ce6f,
    0xb5da05b5, 0x5aea6eac, 0xdca39201, 0x3393f918, 0x072f32c2, 0xe81f59db,
    0xce5c8d57, 0x216ce64e, 0x15d02d94, 0xfae0468d, 0x7ca9ba20, 0x9399d139,
    0xa7251ae3, 0x481571fa, 0xae5a9548, 0x416afe51, 0x75d6358b, 0x9ae65e92,
    0x1cafa23f, 0xf39fc926, 0xc72302fc, 0x281369e5, 0x8ba4abe4, 0x6494c0fd,
    0x50280b27, 0xbf18603e, 0x39519c93, 0xd661f78a, 0xe2dd3c50, 0x0ded5749,
    0xeba2b3fb, 0x0492d8e2, 0x302e1338, 0xdf1e7821, 0x5957848c, 0xb667ef95,
    0x82db244f, 0x6deb4f56, 0x4ba89bda, 0xa498f0c3, 0x90243b19, 0x7f145000,
    0xf95dacad, 0x166dc7b4, 0x22d10c6e, 0xcde16777, 0x2bae83c5, 0xc49ee8dc,
    0xf0222306, 0x1f12481f, 0x995bb4b2, 0x766bdfab, 0x42d71471, 0xade77f68,
    0x0054e682, 0xef648d9b, 0xdbd84641, 0x34e82d58, 0xb2a1d1f5, 0x5d91baec,
    0x692d7136, 0x861d1a2f, 0x6052fe9d, 0x8f629584, 0xbbde5e5e, 0x54ee3547,
    0xd2a7c9ea, 0x3d97a2f3, 0x092b6929, 0xe61b0230, 0xc058d6bc, 0x2f68bda5,
    0x1bd4767f, 0xf4e41d66, 0x72ade1cb, 0x9d9d8ad2, 0xa9214108, 0x46112a11,
    0xa05ecea3, 0x4f6ea5ba, 0x7bd26e60, 0x94e20579, 0x12abf9d4, 0xfd9b92cd,
    0xc9275917, 0x2617320e, 0x85a0f00f, 0x6a909b16, 0x5e2c50cc, 0xb11c3bd5,
    0x3755c778, 0xd865ac61, 0xecd967bb, 0x03e90ca2, 0xe5a6e810, 0x0a968309,
    0x3e2a48d3, 0xd11a23ca, 0x5753df67, 0xb863b47e, 0x8cdf7fa4, 0x63ef14bd,
    0x45acc031, 0xaa9cab28, 0x9e2060f2, 0x71100beb, 0xf759f746, 0x18699c5f,
    0x2cd55785, 0xc3e53c9c, 0x25aad82e, 0xca9ab337, 0xfe2678ed, 0x111613f4,
    0x975fef59, 0x786f8440, 0x4cd34f9a, 0xa3e32483},
   {0xec1416e7, 0x84173a2f, 0x3c124f77, 0x541163bf, 0x49f4d336, 0x21f7fffe,
    0x99f28aa6, 0xf1f1a66e, 0xa239ebb4, 0xca3ac77c, 0x723fb224, 0x1a3c9eec,
    0x07d92e65, 0x6fda02ad, 0xd7df77f5, 0xbfdc5b3d, 0x704fec41, 0x184cc089,
    0xa049b5d1, 0xc84a9919, 0xd5af2990, 0xbdac0558, 0x05a97000, 0x6daa5cc8,
    0x3e621112, 0x56613dda, 0xee644882, 0x8667644a, 0x9b82d4c3, 0xf381f80b,
    0x4b848d53, 0x2387a19b, 0xd14f955a, 0xb94cb992, 0x0149ccca, 0x694ae002,
    0x74af508b, 0x1cac7c43, 0xa4a9091b, 0xccaa25d3, 0x9f626809, 0xf76144c1,
    0x4f643199, 0x27671d51, 0x3a82add8, 0x52818110, 0xea84f448, 0x8287d880,
    0x4d146ffc, 0x25174334, 0x9d12366c, 0xf5111aa4, 0xe8f4aa2d, 0x80f786e5,
    0x38f2f3bd, 0x50f1df75, 0x033992af, 0x6b3abe67, 0xd33fcb3f, 0xbb3ce7f7,
    0xa6d9577e, 0xceda7bb6, 0x76df0eee, 0x1edc2226, 0x96a3119d, 0xfea03d55,
    0x46a5480d, 0x2ea664c5, 0x3343d44c, 0x5b40f884, 0xe3458ddc, 0x8b46a114,
    0xd88eecce, 0xb08dc006, 0x0888b55e, 0x608b9996, 0x7d6e291f, 0x156d05d7,
    0xad68708f, 0xc56b5c47, 0x0af8eb3b, 0x62fbc7f3, 0xdafeb2ab, 0xb2fd9e63,
    0xaf182eea, 0xc71b0222, 0x7f1e777a, 0x171d5bb2, 0x44d51668, 0x2cd63aa0,
    0x94d34ff8, 0xfcd06330, 0xe135d3b9, 0x8936ff71, 0x31338a29, 0x5930a6e1,
    0xabf89220, 0xc3fbbee8, 0x7bfecbb0, 0x13fde778, 0x0e1857f1, 0x661b7b39,
    0xde1e0e61, 0xb61d22a9, 0xe5d56f73, 0x8dd643bb, 0x35d336e3, 0x5dd01a2b,
    0x4035aaa2, 0x2836866a, 0x9033f332, 0xf830dffa, 0x37a36886, 0x5fa0444e,
    0xe7a53116, 0x8fa61dde, 0x9243ad57, 0xfa40819f, 0x4245f4c7, 0x2a46d80f,
    0x798e95d5, 0x118db91d, 0xa988cc45, 0xc18be08d, 0xdc6e5004, 0xb46d7ccc,
    0x0c680994, 0x646b255c, 0x197a1813, 0x717934db, 0xc97c4183, 0xa17f6d4b,
    0xbc9addc2, 0xd499f10a, 0x6c9c8452, 0x049fa89a, 0x5757e540, 0x3f54c988,
    0x8751bcd0, 0xef529018, 0xf2b72091, 0x9ab40c59, 0x22b17901, 0x4ab255c9,
    0x8521e2b5, 0xed22ce7d, 0x5527bb25, 0x3d2497ed, 0x20c12764, 0x48c20bac,
    0xf0c77ef4, 0x98c4523c, 0xcb0c1fe6, 0xa30f332e, 0x1b0a4676, 0x73096abe,
    0x6eecda37, 0x06eff6ff, 0xbeea83a7, 0xd6e9af6f, 0x24219bae, 0x4c22b766,
    0xf427c23e, 0x9c24eef6, 0x81c15e7f, 0xe9c272b7, 0x51c707ef, 0x39c42b27,
    0x6a0c66fd, 0x020f4a35, 0xba0a3f6d, 0xd20913a5, 0xcfeca32c, 0xa7ef8fe4,
    0x1feafabc, 0x77e9d674, 0xb87a6108, 0xd0794dc0, 0x687c3898, 0x007f1450,
    0x1d9aa4d9, 0x75998811, 0xcd9cfd49, 0xa59fd181, 0xf6579c5b, 0x9e54b093,
    0x2651c5cb, 0x4e52e903, 0x53b7598a, 0x3bb47542, 0x83b1001a, 0xebb22cd2,
    0x63cd1f69, 0x0bce33a1, 0xb3cb46f9, 0xdbc86a31, 0xc62ddab8, 0xae2ef670,
    0x162b8328, 0x7e28afe0, 0x2de0e23a, 0x45e3cef2, 0xfde6bbaa, 0x95e59762,
    0x880027eb, 0xe0030b23, 0x58067e7b, 0x300552b3, 0xff96e5cf, 0x9795c907,
    0x2f90bc5f, 0x47939097, 0x5a76201e, 0x32750cd6, 0x8a70798e, 0xe2735546,
    0xb1bb189c, 0xd9b83454, 0x61bd410c, 0x09be6dc4, 0x145bdd4d, 0x7c58f185,
    0xc45d84dd, 0xac5ea815, 0x5e969cd4, 0x3695b01c, 0x8e90c544, 0xe693e98c,
    0xfb765905, 0x937575cd, 0x2b700095, 0x43732c5d, 0x10bb6187, 0x78b84d4f,
    0xc0bd3817, 0xa8be14df, 0xb55ba456, 0xdd58889e, 0x655dfdc6, 0x0d5ed10e,
    0xc2cd6672, 0xaace4aba, 0x12cb3fe2, 0x7ac8132a, 0x672da3a3, 0x0f2e8f6b,
    0xb72bfa33, 0xdf28d6fb, 0x8ce09b21, 0xe4e3b7e9, 0x5ce6c2b1, 0x34e5ee79,
    0x29005ef0, 0x41037238, 0xf9060760, 0x91052ba8},
   {0x3716d8e7, 0x7e2aa5c0, 0xa56e22a9, 0xec525f8e, 0x160b5a8a, 0x5f3727ad,
    0x8473a0c4, 0xcd4fdde3, 0x752ddc3d, 0x3c11a11a, 0xe7552673, 0xae695b54,
    0x54305e50, 0x1d0c2377, 0xc648a41e, 0x8f74d939, 0xb360d153, 0xfa5cac74,
    0x21182b1d, 0x6824563a, 0x927d533e, 0xdb412e19, 0x0005a970, 0x4939d457,
    0xf15bd589, 0xb867a8ae, 0x63232fc7, 0x2a1f52e0, 0xd04657e4, 0x997a2ac3,
    0x423eadaa, 0x0b02d08d, 0x3a16bd7e, 0x732ac059, 0xa86e4730, 0xe1523a17,
    0x1b0b3f13, 0x52374234, 0x8973c55d, 0xc04fb87a, 0x782db9a4, 0x3111c483,
    0xea5543ea, 0xa3693ecd, 0x59303bc9, 0x100c46ee, 0xcb48c187, 0x8274bca0,
    0xbe60b4ca, 0xf75cc9ed, 0x2c184e84, 0x652433a3, 0x9f7d36a7, 0xd6414b80,
    0x0d05cce9, 0x4439b1ce, 0xfc5bb010, 0xb567cd37, 0x6e234a5e, 0x271f3779,
    0xdd46327d, 0x947a4f5a, 0x4f3ec833, 0x0602b514, 0x2d1613d5, 0x642a6ef2,
    0xbf6ee99b, 0xf65294bc, 0x0c0b91b8, 0x4537ec9f, 0x9e736bf6, 0xd74f16d1,
    0x6f2d170f, 0x26116a28, 0xfd55ed41, 0xb4699066, 0x4e309562, 0x070ce845,
    0xdc486f2c, 0x9574120b, 0xa9601a61, 0xe05c6746, 0x3b18e02f, 0x72249d08,
    0x887d980c, 0xc141e52b, 0x1a056242, 0x53391f65, 0xeb5b1ebb, 0xa267639c,
    0x7923e4f5, 0x301f99d2, 0xca469cd6, 0x837ae1f1, 0x583e6698, 0x11021bbf,
    0x2016764c, 0x692a0b6b, 0xb26e8c02, 0xfb52f125, 0x010bf421, 0x48378906,
    0x93730e6f, 0xda4f7348, 0x622d7296, 0x2b110fb1, 0xf05588d8, 0xb969f5ff,
    0x4330f0fb, 0x0a0c8ddc, 0xd1480ab5, 0x98747792, 0xa4607ff8, 0xed5c02df,
    0x361885b6, 0x7f24f891, 0x857dfd95, 0xcc4180b2, 0x170507db, 0x5e397afc,
    0xe65b7b22, 0xaf670605, 0x7423816c, 0x3d1ffc4b, 0xc746f94f, 0x8e7a8468,
    0x553e0301, 0x1c027e26, 0x03174e83, 0x4a2b33a4, 0x916fb4cd, 0xd853c9ea,
    0x220accee, 0x6b36b1c9, 0xb07236a0, 0xf94e4b87, 0x412c4a59, 0x0810377e,
    0xd354b017, 0x9a68cd30, 0x6031c834, 0x290db513, 0xf249327a, 0xbb754f5d,
    0x87614737, 0xce5d3a10, 0x1519bd79, 0x5c25c05e, 0xa67cc55a, 0xef40b87d,
    0x34043f14, 0x7d384233, 0xc55a43ed, 0x8c663eca, 0x5722b9a3, 0x1e1ec484,
    0xe447c180, 0xad7bbca7, 0x763f3bce, 0x3f0346e9, 0x0e172b1a, 0x472b563d,
    0x9c6fd154, 0xd553ac73, 0x2f0aa977, 0x6636d450, 0xbd725339, 0xf44e2e1e,
    0x4c2c2fc0, 0x051052e7, 0xde54d58e, 0x9768a8a9, 0x6d31adad, 0x240dd08a,
    0xff4957e3, 0xb6752ac4, 0x8a6122ae, 0xc35d5f89, 0x1819d8e0, 0x5125a5c7,
    0xab7ca0c3, 0xe240dde4, 0x39045a8d, 0x703827aa, 0xc85a2674, 0x81665b53,
    0x5a22dc3a, 0x131ea11d, 0xe947a419, 0xa07bd93e, 0x7b3f5e57, 0x32032370,
    0x191785b1, 0x502bf896, 0x8b6f7fff, 0xc25302d8, 0x380a07dc, 0x71367afb,
    0xaa72fd92, 0xe34e80b5, 0x5b2c816b, 0x1210fc4c, 0xc9547b25, 0x80680602,
    0x7a310306, 0x330d7e21, 0xe849f948, 0xa175846f, 0x9d618c05, 0xd45df122,
    0x0f19764b, 0x46250b6c, 0xbc7c0e68, 0xf540734f, 0x2e04f426, 0x67388901,
    0xdf5a88df, 0x9666f5f8, 0x4d227291, 0x041e0fb6, 0xfe470ab2, 0xb77b7795,
    0x6c3ff0fc, 0x25038ddb, 0x1417e028, 0x5d2b9d0f, 0x866f1a66, 0xcf536741,
    0x350a6245, 0x7c361f62, 0xa772980b, 0xee4ee52c, 0x562ce4f2, 0x1f1099d5,
    0xc4541ebc, 0x8d68639b, 0x7731669f, 0x3e0d1bb8, 0xe5499cd1, 0xac75e1f6,
    0x9061e99c, 0xd95d94bb, 0x021913d2, 0x4b256ef5, 0xb17c6bf1, 0xf84016d6,
    0x230491bf, 0x6a38ec98, 0xd25aed46, 0x9b669061, 0x40221708, 0x091e6a2f,
    0xf3476f2b, 0xba7b120c, 0x613f9565, 0x2803e842}
};

uint32_t crc32iscsi_byte(uint32_t crc, void const *mem, size_t len) {
    uint8_t const *data = mem;
    if (data == NULL)
        return 0;
    for (size_t i = 0; i < len; i++) {
        crc = (crc >> 8) ^
              table_byte[(crc ^ data[i]) & 0xff];
    }
    return crc;
}

// This code assumes that integers are stored little-endian.

uint32_t crc32iscsi_word(uint32_t crc, void const *mem, size_t len) {
    uint8_t const *data = mem;
    if (data == NULL)
        return 0;
    while (len && ((ptrdiff_t)data & 0x7)) {
        len--;
        crc = (crc >> 8) ^
              table_byte[(crc ^ *data++) & 0xff];
    }
    size_t n = len >> 3;
    for (size_t i = 0; i < n; i++) {
        uint64_t word = crc ^ ((uint64_t const *)data)[i];
        crc = table_word[7][word & 0xff] ^
              table_word[6][(word >> 8) & 0xff] ^
              table_word[5][(word >> 16) & 0xff] ^
              table_word[4][(word >> 24) & 0xff] ^
              table_word[3][(word >> 32) & 0xff] ^
              table_word[2][(word >> 40) & 0xff] ^
              table_word[1][(word >> 48) & 0xff] ^
              table_word[0][word >> 56];
    }
    data += n << 3;
    len &= 7;
    while (len) {
        len--;
        crc = (crc >> 8) ^
              table_byte[(crc ^ *data++) & 0xff];
    }
    return crc;
}

static uint32_t multmodp(uint32_t a, uint32_t b) {
    uint32_t prod = 0;
    for (;;) {
        if (a & 0x80000000) {
            prod ^= b;
            if ((a & 0x7fffffff) == 0)
                break;
        }
        a <<= 1;
        b = b & 1 ? (b >> 1) ^ 0x82f63b78 : b >> 1;
    }
    return prod;
}

static uint32_t const table_comb[] = {
    0x40000000, 0x20000000, 0x08000000, 0x00800000, 0x00008000, 0x82f63b78,
    0x6ea2d55c, 0x18b8ea18, 0x510ac59a, 0xb82be955, 0xb8fdb1e7, 0x88e56f72,
    0x74c360a4, 0xe4172b16, 0x0d65762a, 0x35d73a62, 0x28461564, 0xbf455269,
    0xe2ea32dc, 0xfe7740e6, 0xf946610b, 0x3c204f8f, 0x538586e3, 0x59726915,
    0x734d5309, 0xbc1ac763, 0x7d0722cc, 0xd289cabe, 0xe94ca9bc, 0x05b74f3f,
    0xa51e1f42
};

static uint32_t x8nmodp(uintmax_t n) {
    uint32_t xp = 0x80000000;
    int k = 3;
    for (;;) {
        if (n & 1)
            xp = multmodp(table_comb[k], xp);
        n >>= 1;
        if (n == 0)
            break;
        if (++k == 31)
            k = 0;
    }
    return xp;
}

uint32_t crc32iscsi_comb(uint32_t crc1, uint32_t crc2,
        uintmax_t len2) {
    return multmodp(x8nmodp(len2), crc1) ^ crc2;
}
