/*
 * Copyright (c) 2024, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package sun.security.pkcs;

import sun.security.x509.AlgorithmId;

import java.io.IOException;
import java.io.InvalidObjectException;
import java.io.ObjectInputStream;
import java.io.Serial;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.ProviderException;
import java.security.spec.NamedParameterSpec;
import java.util.Arrays;
import java.util.Objects;

/// Represents a private key from an algorithm family that is specialized
/// with a named parameter set.
///
/// This key is generated by either a [sun.security.provider.NamedKeyPairGenerator]
/// or [sun.security.provider.NamedKeyFactory]. Its [#getAlgorithm] method
/// returns the algorithm family name, while its [#getParams()] method returns
/// the parameter set name as a [NamedParameterSpec] object. The algorithm
/// identifier in the PKCS #8 encoding of the key is always a single OID derived
/// from the parameter set name.
///
/// Besides the existing [PKCS8Key#privKeyMaterial] field, this class optionally
/// supports an expanded format stored in [#expanded]. While `privKeyMaterial`
/// always represents the format used for encoding, `expanded` is always used
/// in computations. The expanded format must be self-sufficient for
/// cryptographic computations without requiring the encoding format.
///
/// 1. If only `privKeyMaterial` is present, it's also the expanded format.
/// 2. If both `privKeyMaterial` and `expanded` are available, `privKeyMaterial`
///    is the encoding format, and `expanded` is the expanded format.
///
/// If the two formats are the same, only `privKeyMaterial` is included, and
/// `expanded` must be `null`. Some implementations might be tempted to put the
/// same value into `privKeyMaterial` and `expanded`. However, problems can
/// arise if they happen to be the same object. To avoid ambiguity, always set
/// `expanded` to `null`.
///
/// If the `expanded` field is required by the algorithm, it is either
/// [calculated from the PKCS #8 encoding][#NamedPKCS8Key(String, byte\[\], Expander)],
/// or [provided directly][#internalCreate(String, String, byte\[\], byte\[\])].
/// In the latter case, the caller must ensure the consistency of the `encoded`
/// and `expanded` arguments. For example, seed and expanded key must match.
///
/// @see sun.security.provider.NamedKeyPairGenerator
public final class NamedPKCS8Key extends PKCS8Key {
    @Serial
    private static final long serialVersionUID = 1L;

    private final String fname;
    private final transient NamedParameterSpec paramSpec;
    private final transient byte[] expanded;

    private transient boolean destroyed = false;

    /// Creates a `NamedPKCS8Key` from raw components.
    ///
    /// @param fname family name
    /// @param pname parameter set name
    /// @param encoded raw key bytes, not null
    /// @param expanded expanded key format, can be `null`.
    private NamedPKCS8Key(String fname, String pname, byte[] encoded, byte[] expanded) {
        this.fname = fname;
        this.paramSpec = new NamedParameterSpec(pname);
        this.expanded = expanded;
        this.privKeyMaterial = Objects.requireNonNull(encoded);
        try {
            this.algid = AlgorithmId.get(pname);
        } catch (NoSuchAlgorithmException e) {
            throw new ProviderException(e);
        }
    }

    /// Creates a `NamedPKCS8Key` from raw components.
    ///
    /// `encoded` and `expanded` won't be cloned, caller must relinquish
    /// ownership. This caller must ensure `encoded` and `expanded` match
    /// each other and `encoded` is valid and internally-consistent.
    ///
    /// @param fname family name
    /// @param pname parameter set name
    /// @param encoded raw key bytes, not null
    /// @param expanded expanded key format, can be `null`.
    public static NamedPKCS8Key internalCreate(String fname, String pname,
            byte[] encoded, byte[] expanded) {
        return new NamedPKCS8Key(fname, pname, encoded, expanded);
    }

    /// Creates a `NamedPKCS8Key` from family name and PKCS #8 encoding.
    ///
    /// @param fname family name
    /// @param encoded PKCS #8 encoding. It is copied so caller can modify
    ///     it after the method call.
    /// @param expander a function that is able to calculate the expanded
    ///     format from the encoding format inside `encoded`. If it recognizes
    ///     the input already in expanded format, it must return `null`.
    ///     This argument must be `null` if the algorithm's expanded format
    ///     is always the same as its encoding format. Whatever the case, the
    ///     ownership of the result is fully granted to this object.
    public NamedPKCS8Key(String fname, byte[] encoded, Expander expander)
            throws InvalidKeyException {
        super(encoded);
        this.fname = fname;
        this.expanded = expander == null
                ? null
                : expander.expand(algid.getName(), this.privKeyMaterial);
        paramSpec = new NamedParameterSpec(algid.getName());
        if (algid.getEncodedParams() != null) {
            throw new InvalidKeyException("algorithm identifier has params");
        }
    }

    @Override
    public String toString() {
        // Do not modify: this can be used by earlier JDKs that
        // do not have the getParams() method
        return paramSpec.getName() + " private key";
    }

    /// Returns the reference to the internal key. Caller must not modify
    /// the content or pass the reference to untrusted application code.
    public byte[] getRawBytes() {
        return privKeyMaterial;
    }

    /// Returns the reference to the key in expanded format. Caller must not
    /// modify the content or pass the reference to untrusted application code.
    public byte[] getExpanded() {
        return expanded == null ? privKeyMaterial : expanded;
    }

    @Override
    public NamedParameterSpec getParams() {
        return paramSpec;
    }

    @Override
    public String getAlgorithm() {
        return fname;
    }

    @java.io.Serial
    private void readObject(ObjectInputStream stream)
            throws IOException, ClassNotFoundException {
        throw new InvalidObjectException(
                "NamedPKCS8Key keys are not directly deserializable");
    }

    @Override
    public void destroy() {
        Arrays.fill(privKeyMaterial, (byte)0);
        if (expanded != null) {
            Arrays.fill(expanded, (byte)0);
        }
        if (encodedKey != null) {
            Arrays.fill(encodedKey, (byte)0);
        }
        destroyed = true;
    }

    @Override
    public boolean isDestroyed() {
        return destroyed;
    }

    /// Expands from encoding format to expanded format.
    @FunctionalInterface
    public interface Expander {
        /// The expand method
        ///
        /// @param pname parameter set name
        /// @param input input encoding
        /// @return the expanded key, `null` if `input` is already in expanded
        /// @throws InvalidKeyException if `input` is invalid, for example,
        ///         wrong encoding, or internal inconsistency
        byte[] expand(String pname, byte[] input) throws InvalidKeyException;
    }
}
