use crate::attrs::AttributeKind;

#[derive(PartialEq)]
pub enum EncodeCrossCrate {
    Yes,
    No,
}

impl AttributeKind {
    /// Whether this attribute should be encoded in metadata files.
    ///
    /// If this is "Yes", then another crate can do `tcx.get_all_attrs(did)` for a did in this crate, and get the attribute.
    /// When this is No, the attribute is filtered out while encoding and other crate won't be able to observe it.
    /// This can be unexpectedly good for performance, so unless necessary for cross-crate compilation, prefer No.
    pub fn encode_cross_crate(&self) -> EncodeCrossCrate {
        use AttributeKind::*;
        use EncodeCrossCrate::*;

        match self {
            // tidy-alphabetical-start
            Align { .. } => No,
            AllowInternalUnsafe(..) => Yes,
            AllowInternalUnstable(..) => Yes,
            AutomaticallyDerived(..) => Yes,
            CfgAttrTrace => Yes,
            CfgTrace(..) => Yes,
            CfiEncoding { .. } => Yes,
            Cold(..) => No,
            CollapseDebugInfo(..) => Yes,
            CompilerBuiltins => No,
            ConstContinue(..) => No,
            Coroutine(..) => No,
            Coverage(..) => No,
            CrateName { .. } => No,
            CrateType(_) => No,
            CustomMir(_, _, _) => Yes,
            DebuggerVisualizer(..) => No,
            Deprecation { .. } => Yes,
            DoNotRecommend { .. } => Yes,
            Doc(_) => Yes,
            DocComment { .. } => Yes,
            EiiDeclaration(_) => Yes,
            EiiForeignItem => No,
            EiiImpls(..) => No,
            ExportName { .. } => Yes,
            ExportStable => No,
            FfiConst(..) => No,
            FfiPure(..) => No,
            Fundamental { .. } => Yes,
            Ignore { .. } => No,
            Inline(..) => No,
            InstructionSet(..) => No,
            Link(..) => No,
            LinkName { .. } => Yes, // Needed for rustdoc
            LinkOrdinal { .. } => No,
            LinkSection { .. } => Yes, // Needed for rustdoc
            Linkage(..) => No,
            LoopMatch(..) => No,
            MacroEscape(..) => No,
            MacroExport { .. } => Yes,
            MacroUse { .. } => No,
            Marker(..) => No,
            MayDangle(..) => No,
            MoveSizeLimit { .. } => No,
            MustNotSupend { .. } => Yes,
            MustUse { .. } => Yes,
            Naked(..) => No,
            NeedsAllocator => No,
            NeedsPanicRuntime => No,
            NoBuiltins => Yes,
            NoCore(..) => No,
            NoImplicitPrelude(..) => No,
            NoLink => No,
            NoMain => No,
            NoMangle(..) => Yes, // Needed for rustdoc
            NoStd(..) => No,
            NonExhaustive(..) => Yes, // Needed for rustdoc
            Optimize(..) => No,
            PanicRuntime => No,
            PatchableFunctionEntry { .. } => Yes,
            Path(..) => No,
            PatternComplexityLimit { .. } => No,
            PinV2(..) => Yes,
            Pointee(..) => No,
            ProcMacro(..) => No,
            ProcMacroAttribute(..) => No,
            ProcMacroDerive { .. } => No,
            ProfilerRuntime => No,
            RecursionLimit { .. } => No,
            Repr { .. } => No,
            RustcAllocator => No,
            RustcAllocatorZeroed => No,
            RustcAllocatorZeroedVariant { .. } => Yes,
            RustcAllowConstFnUnstable(..) => No,
            RustcAllowIncoherentImpl(..) => No,
            RustcAsPtr(..) => Yes,
            RustcBodyStability { .. } => No,
            RustcBuiltinMacro { .. } => Yes,
            RustcCoherenceIsCore(..) => No,
            RustcCoinductive(..) => No,
            RustcConfusables { .. } => Yes,
            RustcConstStability { .. } => Yes,
            RustcConstStabilityIndirect => No,
            RustcDeallocator => No,
            RustcDenyExplicitImpl(..) => No,
            RustcDummy => No,
            RustcDumpDefParents => No,
            RustcDumpItemBounds => No,
            RustcDumpPredicates => No,
            RustcDumpUserArgs => No,
            RustcDumpVtable(..) => No,
            RustcDynIncompatibleTrait(..) => No,
            RustcHasIncoherentInherentImpls => Yes,
            RustcLayoutScalarValidRangeEnd(..) => Yes,
            RustcLayoutScalarValidRangeStart(..) => Yes,
            RustcLegacyConstGenerics { .. } => Yes,
            RustcLintOptDenyFieldAccess { .. } => Yes,
            RustcLintOptTy => Yes,
            RustcLintQueryInstability => Yes,
            RustcLintUntrackedQueryInformation => Yes,
            RustcMacroTransparency(..) => Yes,
            RustcMain => No,
            RustcMustImplementOneOf { .. } => No,
            RustcNeverReturnsNullPointer => Yes,
            RustcNoImplicitAutorefs => Yes,
            RustcNounwind => No,
            RustcObjcClass { .. } => No,
            RustcObjcSelector { .. } => No,
            RustcObjectLifetimeDefault => No,
            RustcOffloadKernel => Yes,
            RustcParenSugar(..) => No,
            RustcPassByValue(..) => Yes,
            RustcPassIndirectlyInNonRusticAbis(..) => No,
            RustcPubTransparent(..) => Yes,
            RustcReallocator => No,
            RustcScalableVector { .. } => Yes,
            RustcShouldNotBeCalledOnConstItems(..) => Yes,
            RustcSimdMonomorphizeLaneLimit(..) => Yes, // Affects layout computation, which needs to work cross-crate
            RustcSkipDuringMethodDispatch { .. } => No,
            RustcSpecializationTrait(..) => No,
            RustcStdInternalSymbol(..) => No,
            RustcUnsafeSpecializationMarker(..) => No,
            RustcVariance => No,
            RustcVarianceOfOpaques => No,
            Sanitize { .. } => No,
            ShouldPanic { .. } => No,
            Stability { .. } => Yes,
            TargetFeature { .. } => No,
            ThreadLocal => No,
            TrackCaller(..) => Yes,
            TypeConst(..) => Yes,
            TypeLengthLimit { .. } => No,
            UnstableFeatureBound(..) => No,
            Used { .. } => No,
            WindowsSubsystem(..) => No,
            // tidy-alphabetical-end
        }
    }
}
