/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

/*
 * dump-info - start of a tool to dump the information on a dump tape for
 * debugging purposes
 */

#include "config.h" // IWYU pragma: keep
#include "faketape-lib.h"

#include "bswap_header.h"

#include <inttypes.h>
#include <sys/types.h>
#include <iostream>
#include <vector>
#include <unordered_map>
#include <cstring>
#include <sys/stat.h>
#include <fcntl.h>
#include <err.h>
#include <unistd.h>
#include <getopt.h>
#include <zlib.h>
#include <bzlib.h>
#include <lzo/lzo1x.h>
#include <ext2fs/ext2_fs.h>
#include <ext2fs/ext2_ext_attr.h>
#include <optional>
#include <endian.h>
#include <lzo/lzoconf.h>
#include <stdio.h>
#include <stdlib.h>
#include <zconf.h>
#include <algorithm>
#include <memory>
#include <string>
#include <string_view>

// Messy. Ideally we'd use readblock_tape from restore/readtape.c but there are
// too many issues with global variables that we don't use but need to link in.

// Manually copied from protocols/dumprestore.h
#define ROOTINO              EXT2_ROOT_INO

/*
 * extattributes inode info
 */
#define EXT_REGULAR		0
#define EXT_MACOSFNDRINFO	1
#define EXT_MACOSRESFORK	2
#define EXT_XATTR		3

#define COMPRESS_ZLIB	0
#define COMPRESS_BZLIB	1
#define COMPRESS_LZO	2

std::string convert_type(uint32_t type) {
#define C(X) case X: return std::string(#X "        ").substr(0,8); break
	switch (type) {
		C(TS_TAPE);
		C(TS_INODE);
		C(TS_ADDR);
		C(TS_BITS);
		C(TS_CLRI);
		C(TS_END);
	default:
		return "unknown " + std::to_string(type);
	}
#undef C
}

std::string convert_flags(int32_t flags) {
#define C(X) if(flags&(X)) { ret += "|" #X; flags &= ~(X); }
	std::string ret;
	C(DR_NEWHEADER);
	C(DR_NEWINODEFMT);
	C(DR_COMPRESSED);
	C(DR_METAONLY);
	C(DR_EXTATTRIBUTES);
	if (flags)
		ret += "|" + std::to_string(flags);
	if(ret.size())
		return ret.substr(1);
	return "0";
#undef C
}

std::vector<uint8_t> databuffer;

[[noreturn]] void usage(const char* prog, int exitcode) {
	auto wrap = [](const std::string& p1, const std::string& msg) {
		std::string m = p1 + msg;
		while (true) {
			auto sp = std::min(m.find("\n"), m.find(" ", 70));
			if (sp == std::string::npos) {
				std::cerr << m << std::endl;
				return;
			}
			std::cerr << m.substr(0, sp) << std::endl;
			m = p1 + m.substr(sp+1);
		}
	};

	std::cerr << "Usage " << prog << " dump-img" << std::endl;
	std::cerr << "  -e, --zero-ea-padding" << std::endl;
	std::cerr << "  -p, --zero-directory-padding" << std::endl;
	std::cerr << "  -s, --zero-symlink-padding" << std::endl;
	wrap("       ",
		"older versions of dump used to write uninitialized data "
		"beyond the end of the data from the inode. These three "
		"flags cause dump-info to zero these bytes before "
		"calculating the crc for the block");
	std::cerr << "  -m, --print-header" << std::endl;
	wrap("       ",
		"print out the detailed information in the dump header block "
		"for the inode. Note that this is done after normalization if "
		"--normalize-header is also used.");
	std::cerr << "  -n, --normalize-header" << std::endl;
	wrap("       ",
		"zero out the bytes in the dump header block for the inode "
		"that are expected differ between dumps of the same fs before "
		"calculating the crc.\nThis also zeros out the h_checksum in "
		"any ext2_ext_attr_header.");
	std::cerr << "  -i, --normalize-inodes" << std::endl;
	wrap("       ",
		"Normalize inodes so that they are assigned in the order "
		"they're seen in the dump rather than whatever numbers they "
		"got in real life. Note that this will only work reliably "
		"where directory inodes are found in order");
	std::cerr << "  -v, --version" << std::endl;
	std::cerr << "  -d, --debug" << std::endl;
	exit(exitcode);
}

[[noreturn]] void fatal(const char* msg, const char* prog) {
	std::cerr << "FATAL: " << msg << std::endl << std::endl;
	usage(prog, 1);
}

std::string getmode(uint16_t mode) {
	std::string result;
	if (mode & S_ISUID) result += 'r'; else result += '-';

	if (mode & S_IRUSR) result += 'r'; else result += '-';
	if (mode & S_IWUSR) result += 'w'; else result += '-';
	if (mode & S_ISGID) {
		if (mode & S_IXUSR) result += 's'; else result += 'S';
	} else
		if (mode & S_IXUSR) result += 'x'; else result += '-';

	if (mode & S_IRGRP) result += 'r'; else result += '-';
	if (mode & S_IWGRP) result += 'w'; else result += '-';

	if (mode & S_ISGID) {
		if (mode & S_IXGRP) result += 's'; else result += 'S';
	} else
		if (mode & S_IXGRP) result += 'x'; else result += '-';

	if (mode & S_IROTH) result += 'r'; else result += '-';
	if (mode & S_IWOTH) result += 'w'; else result += '-';

	if (mode & S_ISVTX) {
		if (mode & S_IXOTH) result += 't'; else result += 'T';
	} else
		if (mode & S_IXOTH) result += 'x'; else result += '-';

	switch(mode & S_IFMT) {
		case S_IFDIR: result += " directory"; break;
		case S_IFCHR: result += " character special file "; break;
		case S_IFBLK: result += " block special file"; break;
		case S_IFREG: result += " regular file"; break;
		case S_IFIFO: result += " FIFO special file"; break;
		case S_IFLNK: result += " symbolic link"; break;
		case S_IFSOCK: result += " socket"; break;
		default:       result += " UNKNOWN"; break;
	}
	return result;
}

uint32_t crc32(const uint8_t* data, size_t size, uint32_t crc) {
	static constexpr uint32_t crcTable[256] = {
		0x00000000, 0xedb88320, 0x36c98560, 0xdb710640, 0x6d930ac0, 0x802b89e0, 0x5b5a8fa0, 0xb6e20c80,
		0xdb261580, 0x369e96a0, 0xedef90e0, 0x005713c0, 0xb6b51f40, 0x5b0d9c60, 0x807c9a20, 0x6dc41900,
		0x5bf4a820, 0xb64c2b00, 0x6d3d2d40, 0x8085ae60, 0x3667a2e0, 0xdbdf21c0, 0x00ae2780, 0xed16a4a0,
		0x80d2bda0, 0x6d6a3e80, 0xb61b38c0, 0x5ba3bbe0, 0xed41b760, 0x00f93440, 0xdb883200, 0x3630b120,
		0xb7e95040, 0x5a51d360, 0x8120d520, 0x6c985600, 0xda7a5a80, 0x37c2d9a0, 0xecb3dfe0, 0x010b5cc0,
		0x6ccf45c0, 0x8177c6e0, 0x5a06c0a0, 0xb7be4380, 0x015c4f00, 0xece4cc20, 0x3795ca60, 0xda2d4940,
		0xec1df860, 0x01a57b40, 0xdad47d00, 0x376cfe20, 0x818ef2a0, 0x6c367180, 0xb74777c0, 0x5afff4e0,
		0x373bede0, 0xda836ec0, 0x01f26880, 0xec4aeba0, 0x5aa8e720, 0xb7106400, 0x6c616240, 0x81d9e160,
		0x826a23a0, 0x6fd2a080, 0xb4a3a6c0, 0x591b25e0, 0xeff92960, 0x0241aa40, 0xd930ac00, 0x34882f20,
		0x594c3620, 0xb4f4b500, 0x6f85b340, 0x823d3060, 0x34df3ce0, 0xd967bfc0, 0x0216b980, 0xefae3aa0,
		0xd99e8b80, 0x342608a0, 0xef570ee0, 0x02ef8dc0, 0xb40d8140, 0x59b50260, 0x82c40420, 0x6f7c8700,
		0x02b89e00, 0xef001d20, 0x34711b60, 0xd9c99840, 0x6f2b94c0, 0x829317e0, 0x59e211a0, 0xb45a9280,
		0x358373e0, 0xd83bf0c0, 0x034af680, 0xeef275a0, 0x58107920, 0xb5a8fa00, 0x6ed9fc40, 0x83617f60,
		0xeea56660, 0x031de540, 0xd86ce300, 0x35d46020, 0x83366ca0, 0x6e8eef80, 0xb5ffe9c0, 0x58476ae0,
		0x6e77dbc0, 0x83cf58e0, 0x58be5ea0, 0xb506dd80, 0x03e4d100, 0xee5c5220, 0x352d5460, 0xd895d740,
		0xb551ce40, 0x58e94d60, 0x83984b20, 0x6e20c800, 0xd8c2c480, 0x357a47a0, 0xee0b41e0, 0x03b3c2c0,
		0xe96cc460, 0x04d44740, 0xdfa54100, 0x321dc220, 0x84ffcea0, 0x69474d80, 0xb2364bc0, 0x5f8ec8e0,
		0x324ad1e0, 0xdff252c0, 0x04835480, 0xe93bd7a0, 0x5fd9db20, 0xb2615800, 0x69105e40, 0x84a8dd60,
		0xb2986c40, 0x5f20ef60, 0x8451e920, 0x69e96a00, 0xdf0b6680, 0x32b3e5a0, 0xe9c2e3e0, 0x047a60c0,
		0x69be79c0, 0x8406fae0, 0x5f77fca0, 0xb2cf7f80, 0x042d7300, 0xe995f020, 0x32e4f660, 0xdf5c7540,
		0x5e859420, 0xb33d1700, 0x684c1140, 0x85f49260, 0x33169ee0, 0xdeae1dc0, 0x05df1b80, 0xe86798a0,
		0x85a381a0, 0x681b0280, 0xb36a04c0, 0x5ed287e0, 0xe8308b60, 0x05880840, 0xdef90e00, 0x33418d20,
		0x05713c00, 0xe8c9bf20, 0x33b8b960, 0xde003a40, 0x68e236c0, 0x855ab5e0, 0x5e2bb3a0, 0xb3933080,
		0xde572980, 0x33efaaa0, 0xe89eace0, 0x05262fc0, 0xb3c42340, 0x5e7ca060, 0x850da620, 0x68b52500,
		0x6b06e7c0, 0x86be64e0, 0x5dcf62a0, 0xb077e180, 0x0695ed00, 0xeb2d6e20, 0x305c6860, 0xdde4eb40,
		0xb020f240, 0x5d987160, 0x86e97720, 0x6b51f400, 0xddb3f880, 0x300b7ba0, 0xeb7a7de0, 0x06c2fec0,
		0x30f24fe0, 0xdd4accc0, 0x063bca80, 0xeb8349a0, 0x5d614520, 0xb0d9c600, 0x6ba8c040, 0x86104360,
		0xebd45a60, 0x066cd940, 0xdd1ddf00, 0x30a55c20, 0x864750a0, 0x6bffd380, 0xb08ed5c0, 0x5d3656e0,
		0xdcefb780, 0x315734a0, 0xea2632e0, 0x079eb1c0, 0xb17cbd40, 0x5cc43e60, 0x87b53820, 0x6a0dbb00,
		0x07c9a200, 0xea712120, 0x31002760, 0xdcb8a440, 0x6a5aa8c0, 0x87e22be0, 0x5c932da0, 0xb12bae80,
		0x871b1fa0, 0x6aa39c80, 0xb1d29ac0, 0x5c6a19e0, 0xea881560, 0x07309640, 0xdc419000, 0x31f91320,
		0x5c3d0a20, 0xb1858900, 0x6af48f40, 0x874c0c60, 0x31ae00e0, 0xdc1683c0, 0x07678580, 0xeadf06a0,
	};

	for (size_t i = 0; i < size; ++i) {
		const uint32_t lookupIndex = (crc ^ data[i]) & 0xff;
		crc = (crc >> 8) ^ crcTable[lookupIndex];
	}

	if (!size) {
		// Finalize the CRC-32 value by inverting all the bits
		crc ^= 0xFFFFFFFFu;
	}
	return crc;
}

static bool print_header = false;
static bool normalize_inodes = false;
static bool normalize_header = false;
static bool zero_symlink_padding = false;
static bool zero_directory_padding = false;
static bool zero_ea_padding = false;
static std::unordered_map<dump_ino_t, std::string> inode_map;

std::string ino_to_string(dump_ino_t ino) {
	if (inode_map.contains(ino))
		return inode_map[ino];
	else
		return std::to_string(ino);
}

void print_tape(uint8_t* data, size_t bytes, bswap_context* ctx) {
	static bool inode_is_ea = false;
	static bool inode_is_symlink = false;
	static std::string indent = "";
	static size_t blockno = 1;
	static uint32_t crc;

	if(!data || !bytes) {
		if(ctx->seen_end) {
			ctx->subcount = 0;
			ctx->state = 0;
			ctx->count = 0;
			ctx->inode_is_directory = false;
			ctx->inode_data_size = 0;
			indent = "";
			blockno = 1;
			if (normalize_inodes) {
				inode_map.clear();
				inode_map[ROOTINO] = "/";
			}
		} else
			ctx->new_vol = true;
		return;
	}
	if (bytes % sizeof(header) == 4) {
		bytes -= 4;
		data += 4;
	}
	header* hdr = reinterpret_cast<header*>(data);
	bytes /= sizeof *hdr;

	auto skip_holes = [&ctx]() {
		static size_t data_indexes = 0;
		static size_t data_blocks = 0;
		size_t holes_skipped = 0;
		size_t indexes = 0;
		while (ctx->count && !(*ctx->s_addr_ptr&1)) {
			holes_skipped += get_s_addr_length(ctx->s_addr_ptr);
			size_t hole_size = get_s_addr_length(ctx->s_addr_ptr) * 1024;
			if (hole_size > ctx->inode_data_size)
				hole_size = ctx->inode_data_size;
			ctx->inode_data_size -= hole_size;
			indexes++;
			if(get_s_addr_length(ctx->s_addr_ptr) > 64)
				ctx->s_addr_ptr += 2;
			else
				ctx->s_addr_ptr++;
			ctx->count--;
		}
		if (!ctx->count) {
			if (data_blocks) {
				crc = crc32(nullptr, 0, crc);
				std::cout << indent << "    Processed " << data_blocks << " data blocks in " << data_indexes << " indexes" << " crc=" << std::hex << crc << std::dec << std::endl;
			}
			ctx->subcount = 0;
			ctx->state = 0;
			data_indexes = 0;
			data_blocks = 0;
		} else {
			ctx->subcount = get_s_addr_length(ctx->s_addr_ptr);
			data_blocks += ctx->subcount;
			data_indexes++;
		}
		if (holes_skipped)
			std::cout << indent << "Skipped " << holes_skipped << " holes in " << indexes << " indexes" << std::endl;
	};
#define EXT2_XATTR_MAGIC		0xEA020000	// block EA
#define EXT2_XATTR_MAGIC2		0xEA020001	// in inode EA

	auto print_ea = [&]() {
		static std::vector<uint8_t> eabuffer;
		if (ctx->count) {
			if (!eabuffer.size()) {
				/* h_checksum includes the blocknum in the
				 * calculation so it's useless in the dump and
				 * causes dumps to be different just based on
				 * the block that the EAs were stored in on the
				 * disk */
				ext2_ext_attr_header* header = reinterpret_cast<ext2_ext_attr_header*>(hdr);
				if (tohost(header->h_magic) == EXT2_EXT_ATTR_MAGIC && normalize_header) {
					header->h_blocks = 0;	// h_blocks and h_hash appear to not be set when using fuse
					header->h_hash = 0;
					header->h_checksum = 0;
				}
			}
			eabuffer.insert(eabuffer.end(), reinterpret_cast<uint8_t*>(hdr), reinterpret_cast<uint8_t*>(hdr)+sizeof *hdr);
			return;
		}
		const uint8_t* data_start;
		const ext2_ext_attr_entry* entry;
		const ext2_ext_attr_header* header = reinterpret_cast<const ext2_ext_attr_header*>(eabuffer.data());
		switch(tohost(header->h_magic)) {
			case EXT2_EXT_ATTR_MAGIC:
				data_start = reinterpret_cast<const uint8_t*>(header);
				entry = reinterpret_cast<const ext2_ext_attr_entry*>(header+1);
				break;
			case EXT2_EXT_ATTR_MAGIC+1:
				data_start = reinterpret_cast<const uint8_t*>(header) + sizeof header->h_magic;
				entry = reinterpret_cast<const ext2_ext_attr_entry*>(data_start);
				break;
			default:
				errx(1, "Unknown magic %x in print_ea", header->h_magic);
		}
		while(tohost(entry->e_name_len) || tohost(entry->e_name_index) || tohost(entry->e_value_offs)) {
			std::cout << "entry->e_name_len=" << (int)tohost(entry->e_name_len) << " entry->e_name_index=" << (int)tohost(entry->e_name_index) << " entry->e_value_offs=" << tohost(entry->e_value_offs) << std::endl;
			std::string_view nametxt{ reinterpret_cast<const char*>(entry)+sizeof *entry, tohost(entry->e_name_len)};
			std::string name;
			switch (tohost(entry->e_name_index)) {
				case 10:
					name = std::string("gnu.") + std::string(nametxt);
					break;
				case 3:
					name = std::string("system.posix_acl_default.") + std::string(nametxt);
					break;
				case 2:
					name = std::string("system.posix_acl_access.") + std::string(nametxt);
					break;
				case 8:
					name = std::string("system.richacl.") + std::string(nametxt);
					break;
				case 6:
					name = std::string("security.") + std::string(nametxt);
					break;
				case 4:
					name = std::string("trusted.") + std::string(nametxt);
					break;
				case 7:
					name = std::string("system.") + std::string(nametxt);
					break;
				case 1:
					name = std::string("user.") + std::string(nametxt);
					break;
				default:
					errx(1, "unknown attribute");
			}
			std::vector<uint8_t> value;
			value.insert(value.end(), data_start + tohost(entry->e_value_offs), data_start + tohost(entry->e_value_offs) + tohost(entry->e_value_size));
			std::cout << indent << "    EA: " << name << " value size is " << value.size() << std::endl;

			size_t size = ((sizeof *entry + tohost(entry->e_name_len) + EXT2_EXT_ATTR_PAD - 1) / EXT2_EXT_ATTR_PAD) * EXT2_EXT_ATTR_PAD;
			entry = reinterpret_cast<const ext2_ext_attr_entry*>(reinterpret_cast<const uint8_t*>(entry) + size);
		}
		eabuffer.clear();
	};

	auto print_direct = [&]() {
		uint8_t* data = reinterpret_cast<uint8_t*>(hdr);
		size_t consumed = 0;
		direct* ptr = reinterpret_cast<direct*>(data);

		auto strtype = [](uint8_t type) -> std::string {
			switch((uint32_t)type << 12) {
				case S_IFDIR: return "directory"; break;
				case S_IFCHR: return "character special file "; break;
				case S_IFBLK: return "block special file"; break;
				case S_IFREG: return "regular file"; break;
				case S_IFIFO: return "FIFO special file"; break;
				case S_IFLNK: return "symbolic link"; break;
				case S_IFSOCK: return "socket"; break;
				case 0: return "UNKNOWN:0"; break;
				// Versions 0.4b30 and 0.4b31 had a bug where this wasn't initialized.
				default:       return "UNKNOWN" + std::to_string(type);  break;
			}
		};
		while(consumed < sizeof *hdr && ctx->inode_data_size) {
			uint8_t type = ptr->d_type;
			uint8_t namlen = ptr->d_namlen;
			uint16_t reclen = ptr->d_reclen;
			if (zero_directory_padding && sizeof(direct) + namlen < reclen) {
				memset(data+sizeof(direct)+namlen, 0, reclen-sizeof(direct)-namlen);
			}
			if (normalize_inodes) {
				if (!inode_map.contains(ptr->d_ino)) {
					if (std::string(".") == ptr->d_name || std::string("..") == ptr->d_name)
						inode_map[ptr->d_ino] = "UNKNOWN(" + std::to_string(ptr->d_ino) + ")/";
					else if (inode_map.contains(ctx->inode_num))
						inode_map[ptr->d_ino] = inode_map[ctx->inode_num] + ptr->d_name + (((uint32_t)type << 12)==S_IFDIR?"/":"");
					else	// should never get here?
						inode_map[ptr->d_ino] = "UNKNOWN(" + std::to_string(ctx->inode_num) + ")/" + ptr->d_name + (((uint32_t)type << 12)==S_IFDIR?"/":"");
				}
			}
			std::cout << indent << "  ino=" << ino_to_string(ptr->d_ino) << " reclen=" << ptr->d_reclen << " type=" << strtype(type) << " namlen=" << (int)namlen << " : " << ptr->d_name << std::endl;
			if (normalize_inodes)
				ptr->d_ino = 0;
#if BYTE_ORDER == BIG_ENDIAN
			swap4(1, data);	// ino
			swap2(1, data);	// reclen
			if (ctx->oldinofmt) {
				swap2(1, data);	// namelen (was short prior to d_type being added
				data -= 2;
			}
			data -= 6;
#endif
			data += reclen;
			consumed += reclen;
			ctx->inode_data_size -= reclen;
			ptr = reinterpret_cast<direct*>(data);
		}
	};

	auto process_block = [&]() {
		switch(ctx->state) {
			case 0: {
				indent = "";
				if (hdr->c_magic != NFS_MAGIC || !checksum(hdr)) {
					std::cout << "SKIPPING UNKNOWN at blockno " << blockno <<  " hdr->c_magic=" << hdr->c_magic << "NFS_MAGIC=" << NFS_MAGIC << " csum=" << checksum(hdr) << std::endl;
					hdr++;
					blockno++;
					bytes--;
					ctx->subcount = 0;
					ctx->count = 0;
					ctx->inode_is_directory = false;
					ctx->inode_data_size = 0;
					break;
				}
				header normhdr{*hdr};
				if (normalize_header) {
					normhdr.c_date = 0;
					normhdr.c_ddate = 0;
					normhdr.c_checksum = 0;
					memset(normhdr.c_filesys, 0, sizeof normhdr.c_filesys);
					memset(normhdr.c_dev, 0, sizeof normhdr.c_dev);
					memset(normhdr.c_host, 0, sizeof normhdr.c_host);
					normhdr.c_dinode.di_atime.tv_sec = 0;
					normhdr.c_dinode.di_atime.tv_usec = 0;
					normhdr.c_dinode.di_mtime.tv_sec = 0;
					normhdr.c_dinode.di_mtime.tv_usec = 0;
					normhdr.c_dinode.di_ctime.tv_sec = 0;
					normhdr.c_dinode.di_ctime.tv_usec = 0;
					normhdr.c_dinode.di_gen = 0;
					if ((normhdr.c_dinode.di_mode & S_IFMT) != S_IFBLK && (normhdr.c_dinode.di_mode & S_IFMT) != S_IFCHR && !(normhdr.c_dinode.di_flags & EXT4_INLINE_DATA_FL)) {
						memset(normhdr.c_dinode.di_db, 0, sizeof normhdr.c_dinode.di_db);
						memset(normhdr.c_dinode.di_ib, 0, sizeof normhdr.c_dinode.di_ib);
					}
				}
				if (normalize_inodes) {
					normhdr.c_inumber = 0;
					if (hdr->c_type == TS_END || hdr->c_type == TS_TAPE)
						memset(normhdr.c_data.s_inos, 0, sizeof normhdr.c_data.s_inos);
				}
				header crchdr{normhdr};
#if BYTE_ORDER == BIG_ENDIAN
				do_bswap_hdr(&crchdr);
#endif
				if (print_header) {
					fprintf(stderr, "c_type = %d\n", normhdr.c_type);
					fprintf(stderr, "c_date = %d\n", normhdr.c_date);
					fprintf(stderr, "c_ddate = %d\n", normhdr.c_ddate);
					fprintf(stderr, "c_volume = %d\n", normhdr.c_volume);
					fprintf(stderr, "c_tapea_lo = %d\n", normhdr.c_tapea_lo);
					fprintf(stderr, "c_inumber = %d\n", normhdr.c_inumber);
					fprintf(stderr, "c_magic = %d\n", normhdr.c_magic);
					fprintf(stderr, "c_checksum = %d\n", normhdr.c_checksum);

					//fprintf(stderr, "c_dinode = %d\n", normhdr.c_dinode);
					crc = 0xffffffff;
					crc = crc32(reinterpret_cast<const uint8_t*>(&crchdr.c_dinode), sizeof crchdr.c_dinode, crc);
					std::cout << indent << "c_dinode crc=" << std::hex << crc << std::dec << std::endl;
					fprintf(stderr, "c_dinode.di_mode = %d\n", normhdr.c_dinode.di_mode);
					fprintf(stderr, "c_dinode.di_nlink = %d\n", normhdr.c_dinode.di_nlink);
					fprintf(stderr, "c_dinode.di_u.inumber = %d\n", normhdr.c_dinode.di_u.inumber);
					fprintf(stderr, "c_dinode.di_size = %" PRIu64 "\n", normhdr.c_dinode.di_size);
					fprintf(stderr, "c_dinode.di_atime = %d\n", *(uint32_t*)&normhdr.c_dinode.di_atime);
					fprintf(stderr, "c_dinode.di_mtime = %d\n", *(uint32_t*)&normhdr.c_dinode.di_mtime);
					fprintf(stderr, "c_dinode.di_ctime = %d\n", *(uint32_t*)&normhdr.c_dinode.di_ctime);
					//fprintf(stderr, "c_dinode.di_db[NDADDR] = %d\n", normhdr.c_dinode.di_db[NDADDR]);
					//fprintf(stderr, "c_dinode.di_ib[NIADDR] = %d\n", normhdr.c_dinode.di_ib[NIADDR]);
					crc = 0xffffffff;
					crc = crc32(reinterpret_cast<const uint8_t*>(&crchdr.c_dinode.di_db), sizeof crchdr.c_dinode.di_db, crc);
					std::cout << indent << "c_dinode.di_db crc=" << std::hex << crc << std::dec << std::endl;
					crc = 0xffffffff;
					crc = crc32(reinterpret_cast<const uint8_t*>(&crchdr.c_dinode.di_ib), sizeof crchdr.c_dinode.di_ib, crc);
					std::cout << indent << "c_dinode.di_ib crc=" << std::hex << crc << std::dec << std::endl;
					fprintf(stderr, "c_dinode.di_flags = %d\n", normhdr.c_dinode.di_flags);
					fprintf(stderr, "c_dinode.di_blocks = %d\n", normhdr.c_dinode.di_blocks);
					fprintf(stderr, "c_dinode.di_gen = %d\n", normhdr.c_dinode.di_gen);
					fprintf(stderr, "c_dinode.di_uid = %d\n", normhdr.c_dinode.di_uid);
					fprintf(stderr, "c_dinode.di_gid = %d\n", normhdr.c_dinode.di_gid);
					//fprintf(stderr, "c_dinode.di_spare[2] = %d\n", normhdr.c_dinode.di_spare[2]);
					crc = 0xffffffff;
					crc = crc32(reinterpret_cast<const uint8_t*>(&crchdr.c_dinode.di_spare), sizeof crchdr.c_dinode.di_spare, crc);
					std::cout << indent << "c_dinode.spare crc=" << std::hex << crc << std::dec << std::endl;

					fprintf(stderr, "c_count = %d\n", normhdr.c_count);
					//fprintf(stderr, "c_data = %d\n", normhdr.c_data);
					crc = 0xffffffff;
					crc = crc32(reinterpret_cast<const uint8_t*>(&crchdr.c_data), sizeof crchdr.c_data, crc);
					std::cout << indent << "c_data crc=" << std::hex << crc << std::dec << std::endl;
					fprintf(stderr, "c_label[LBLSIZE] = %*.*s\n", LBLSIZE, LBLSIZE, normhdr.c_label);
					fprintf(stderr, "c_level = %d\n", normhdr.c_level);
					fprintf(stderr, "c_filesys[NAMELEN] = %*.*s\n", NAMELEN, NAMELEN, normhdr.c_filesys);
					fprintf(stderr, "c_dev[NAMELEN] = %*.*s\n", NAMELEN, NAMELEN, normhdr.c_dev);
					fprintf(stderr, "c_host[NAMELEN] = %*.*s\n", NAMELEN, NAMELEN, normhdr.c_host);
					fprintf(stderr, "c_flags = %d\n", normhdr.c_flags);
					fprintf(stderr, "c_firstrec_lo = %d\n", normhdr.c_firstrec_lo);
					fprintf(stderr, "c_ntrec = %d\n", normhdr.c_ntrec);
					fprintf(stderr, "c_extattributes = %d\n", normhdr.c_extattributes);
					fprintf(stderr, "c_tapea_hi = %d\n", normhdr.c_tapea_hi);
					fprintf(stderr, "c_firstrec_hi = %d\n", normhdr.c_firstrec_hi);
					//fprintf(stderr, "c_spare[28] = %d\n", normhdr.c_spare);
					crc = 0xffffffff;
					crc = crc32(reinterpret_cast<const uint8_t*>(&crchdr.c_spare), sizeof crchdr.c_spare, crc);
					std::cout << indent << "c_spare crc=" << std::hex << crc << std::dec << std::endl;

				}
				crc = 0xffffffff;
#if BYTE_ORDER == BIG_ENDIAN
				if (hdr->c_type == TS_END || hdr->c_type == TS_TAPE) {
					uint8_t* data = reinterpret_cast<uint8_t*>(crchdr.c_data.s_inos);
					swap4(128, data);
				}
#endif
				crc = crc32(reinterpret_cast<const uint8_t*>(&crchdr), sizeof crchdr, crc);
				std::cout << indent << convert_type(hdr->c_type) << " at blockno " << blockno << " inode=" << ino_to_string(hdr->c_inumber) << " mode=" << getmode(hdr->c_dinode.di_mode) << " size=" << hdr->c_dinode.di_size << " flags=" << convert_flags(hdr->c_flags) << " crc=" << std::hex << crc << std::dec << std::endl;
				crc = 0xffffffff;
				indent = "  ";
				ctx->seen_end = false;
				switch (hdr->c_type) {
					case TS_END:
						ctx->seen_end = true;
						[[fallthrough]];
					case TS_TAPE: {
						if (hdr->c_data.s_inos[1] == ROOTINO)
							for (int i=1; i<128; i++)
								if (hdr->c_data.s_inos[i])
									std::cout << indent << "volinfo[" << i << "] = " << ino_to_string(hdr->c_data.s_inos[i]) << std::endl;
						break;
					}
					case TS_CLRI:
						ctx->count = hdr->c_count;
						ctx->state = ctx->count?TS_CLRI:0;
						std::cout << indent << ctx->count << " data blocks with CLR-inode bitmap ";
						break;
					case TS_BITS:
						ctx->count = hdr->c_count;
						ctx->state = ctx->count?TS_BITS:0;
						std::cout << indent << ctx->count << " data blocks with HAS-inode bitmap ";
						break;
					case TS_INODE:
						std::cout << indent << "INODE " << ino_to_string(hdr->c_inumber) << " mode=" << getmode(hdr->c_dinode.di_mode) << " size=" << hdr->c_dinode.di_size << " flags=" << convert_flags(hdr->c_flags) << std::endl;
						inode_is_ea = hdr->c_flags & DR_EXTATTRIBUTES;
						ctx->inode_is_directory = !inode_is_ea && (hdr->c_dinode.di_mode & S_IFMT) == S_IFDIR;
						inode_is_symlink = !inode_is_ea && (hdr->c_dinode.di_mode & S_IFMT) == S_IFLNK;
						ctx->inode_data_size = hdr->c_dinode.di_size;
						ctx->inode_num = hdr->c_inumber;
						[[fallthrough]];
					case TS_ADDR:
						ctx->count = hdr->c_count;
						static_assert(sizeof ctx->s_addr == sizeof hdr->c_data.s_addrs);
						std::memcpy(ctx->s_addr, hdr->c_data.s_addrs, sizeof ctx->s_addr);

#if 0
						if (inode_is_ea) {
							for(size_t i = 0; i<ctx->count; ++i) {
								if(ctx->s_addr[i] >= 2)
									break;	/* must be 0.4b49 or later */
								else if(ctx->s_addr[i] == 0) {
									std::cerr << "Fixing up invalid c_addr record for xattrs (v0.4b40-0.4b42 bug)\n";
									for(size_t i = 0; i<ctx->count; ++i)
										ctx->s_addr[i] = 1;
									break;
								}
							}
						}
#endif
						ctx->s_addr_ptr = ctx->s_addr;
						skip_holes();
						if (ctx->count)
							ctx->state = TS_INODE;
						break;
					default:
						errx(1, "unknown state on tape");
						break;
				}
				if (normalize_inodes)
					hdr->c_inumber = 0;
				hdr++;
				blockno++;
				bytes--;
				break;
			}
			case TS_INODE: {
				if (ctx->inode_data_size < sizeof *hdr) {
					if ((zero_symlink_padding && inode_is_symlink) ||
							(zero_directory_padding && ctx->inode_is_directory) ||
							(zero_ea_padding && inode_is_ea)) {
						memset(reinterpret_cast<uint8_t*>(hdr)+ctx->inode_data_size, 0, sizeof *hdr - ctx->inode_data_size);
					}
				}

				if (ctx->inode_is_directory)
					print_direct();
				else if (inode_is_ea)
					print_ea();
				header crchdr{*hdr};
				crc = crc32(reinterpret_cast<const uint8_t*>(&crchdr), sizeof crchdr, crc);
				hdr++;
				blockno++;
				bytes--;
				if(--ctx->subcount == 0) {
					if(--ctx->count == 0) {
						ctx->state = 0;
						if (inode_is_ea)
							print_ea();
					} else {
						if(get_s_addr_length(ctx->s_addr_ptr) > 64)
							ctx->s_addr_ptr += 2;
						else
							ctx->s_addr_ptr++;
					}
					skip_holes();
				}
				break;
				}
			case TS_BITS:
			case TS_CLRI: {
				header crchdr{*hdr};
				if (normalize_inodes)
					memset(&crchdr, 0, sizeof crchdr);
				crc = crc32(reinterpret_cast<const uint8_t*>(&crchdr), sizeof crchdr, crc);
				hdr++;
				blockno++;
				bytes--;
				if(--ctx->count == 0) {
					ctx->state = 0;
					std::cout << "crc=" << std::hex << crc << std::dec << std::endl;
				}
				break;
				}
			default:
				errx(1, "unexpected state while reading tape");
				break;
		}
	};
	if (ctx->new_vol) {
		if (hdr->c_magic != NFS_MAGIC || !checksum(hdr) || hdr->c_type != TS_TAPE) {
			std::cout << "EXPECTED TS_TAPE at start of volume hdr->c_magic=" << std::hex << hdr->c_magic << std::endl;
		} else {
			int saved_state = ctx->state;
			ctx->state = 0;
			process_block();
			ctx->state = saved_state;
		}
		ctx->new_vol = false;
	}
	while (bytes)
		process_block();
}

int main(int argc, char* argv[]) {

	FakeTape	tapein;
	FILE*		filein = nullptr;
	const char* prog = argv[0];

	const char* opts = "dehimnpsv";
	struct option lopts[] = {
		{ .name = "debug", .has_arg = 0, .flag = nullptr, .val = 'd' },
		{ .name = "zero-ea-padding", .has_arg = 0, .flag = nullptr, .val = 'e' },
		{ .name = "normalize-inodes", .has_arg = 0, .flag = nullptr, .val = 'i' },
		{ .name = "print-header", .has_arg = 0, .flag = nullptr, .val = 'm' },
		{ .name = "normalize-header", .has_arg = 0, .flag = nullptr, .val = 'n' },
		{ .name = "zero-directory-padding", .has_arg = 0, .flag = nullptr, .val = 'p' },
		{ .name = "zero-symlink-padding", .has_arg = 0, .flag = nullptr, .val = 's' },
		{ .name = "version", .has_arg = 0, .flag = nullptr, .val = 'v' },
		{},
	};
	std::string tapedev;

	int ch;

	while (( ch = getopt_long(argc, argv, opts, lopts, nullptr)) != -1)
		switch(ch) {
			case 'd':
				tapein.debug = std::make_unique<std::ostream>(std::cerr.rdbuf());
				tapein.debug->copyfmt(std::cerr);
				tapein.debug->clear(std::cerr.rdstate());
				break;
			case 'e':
				zero_ea_padding = true;
				break;
			case 'i':
				normalize_inodes = true;
				inode_map[ROOTINO] = "/";
				break;
			case 'm':
				print_header = true;
				break;
			case 'n':
				normalize_header = true;
				break;
			case 'p':
				zero_directory_padding = true;
				break;
			case 's':
				zero_symlink_padding = true;
				break;
			case 'v':
				std::cerr << prog << " version " _DUMP_VERSION << std::endl;
				return 0;
			case 'h':
				usage(prog, 0);
			default:
				usage(prog, 1);
		}
	argc -= optind;
	argv += optind;

	if (argc < 1)
		usage(prog, 1);

	bswap_context inctx;
	bswap_context printctx;
	inctx.to_host = true;

	while(argc) {

		warnx("Starting %s %s", prog, argv[0]);

		/* Now open the dump image */
		if(tapein.open(argv[0], O_RDONLY) != FakeTape::Response::OK)
			if(!(filein = fopen(argv[0], "r")))
				err(1, "%s cannot be opened.",  argv[0]);

		std::vector<uint8_t> buffer(FakeTape::getMaxBlockSize());

		uint32_t ntrec = 0;
		bool compressed = false;

		auto rewind = [&tapein, &filein]() -> void {
			if(filein) {
				if (fseek(filein, -1024, SEEK_CUR) < 0)
					err(1, "Failed to rewind dump image");
			}
			else if(tapein.bsr(1) != FakeTape::Response::OK)
				err(1, "Failed to rewind tape image");
		};

		auto readData = [&tapein, &filein, &ntrec](std::vector<uint8_t>& buffer) -> size_t {
			size_t readlen;
			if(filein) {
				size_t readsize = std::min(buffer.size(), (ntrec?ntrec:1)*(size_t)1024);
				readlen = fread(buffer.data(), 1, readsize, filein);
			}
			else if(tapein.readData(buffer.data(), buffer.size(), &readlen) != FakeTape::Response::OK)
				err(1, "Failed to read block from dump image");
			return readlen;
		};

		auto readExtra = [&tapein, &filein](uint8_t* buffer) -> size_t {
			size_t readlen;
			if(filein) {
				readlen = fread(buffer, 1, 4, filein);
			}
			else if(tapein.readData(buffer, 4, &readlen) != FakeTape::Response::OK)
				err(1, "Failed to read continuation block from dump image");
			return readlen;
		};


		if (lzo_init() != LZO_E_OK) {
			fatal("internal error - lzo_init failed \n", prog);
		}

		do {
			/* read the first 1K of bytes from the tape */
			bswap_context tmp_inctx = inctx;
			bswap_header(nullptr, 0, &tmp_inctx);
			ntrec = 0;
			/* Read the first block */
			size_t readlen = readData(buffer);
			if (readlen == 0)
				break;
			if (readlen < sizeof(header))
				err(1, "Incomplete block read");

			header* hdr = reinterpret_cast<header*>(buffer.data());

			if (!checksum(hdr))
				errx(1, "This doesn't appear to be a dump tape, or was written on a system with byteswapped values");

			if (hdr->c_magic != NFS_MAGIC) {
				tmp_inctx.to_cpu = true;
				warnx("Input tape appears to be byteswapped");
			}
			bswap_header(buffer.data(), readlen, &tmp_inctx);

			if (hdr->c_magic != NFS_MAGIC)
				errx(1, "magic failure");

			compressed = hdr->c_flags & DR_COMPRESSED;
			ntrec = hdr->c_ntrec;

			std::cout << "Tape has ntrec of " << ntrec << std::endl;
			rewind();

			/* Now start properly */
			inctx.to_cpu = tmp_inctx.to_cpu;
			bswap_header(nullptr, 0, &inctx);
			print_tape(nullptr, 0, &printctx);
			readlen = readData(buffer);
			if((ntrec && readlen != ntrec*1024) || readlen%1024)
				errx(1, "Bad blocksize read on first block %zu", readlen);

			bswap_header(buffer.data(), readlen, &inctx);
			print_tape(buffer.data(), readlen, &printctx);

			std::optional<bool> demangle_bitfields;
			while(true) {

				readlen = readData(buffer);
				if (readlen == 0)
					break;
				uint32_t bsize = readlen;
				uint32_t ctype = 0;
				uint32_t cflag = 0;

				if (compressed) {
					if (inctx.to_cpu) {
						auto data = buffer.data();
						swap4(1, data);
					}
					uint32_t clen = *reinterpret_cast<uint32_t*>(buffer.data());
					if (!demangle_bitfields.has_value()) {
						if(clen>>31)
							// Case 2b. Only way mc can be set.
							demangle_bitfields = true;
						/* blocks written without compression must be exactly ntrec*TP_BSIZE in size */
						else if ((clen & 0xfffffff) != ntrec * TP_BSIZE)
							// This covers case 2a and 1a. 2a because length will be odd when read mangled, and 1a because length will be 16x too big.
							demangle_bitfields = false;
						else
							// Only case 1b remains.
							demangle_bitfields = true;
						if (*demangle_bitfields)
							std::cout << "Warning: Tape appears to have been written with different bitfield ordering" << std::endl;
					}
					if (*demangle_bitfields) {
						bsize = clen & 0xfffffff;
						ctype = (clen&0x70000000)>>28;
						cflag = clen>>31;
					} else {
						bsize = clen >> 4;
						ctype = (clen&0xe)>>1;
						cflag = clen&1;
					}
					if (filein) {
						if (bsize < readlen) {
							fseek(filein, (off_t)bsize+4-(off_t)readlen, SEEK_CUR);
							readlen = bsize + 4;
						}
					}
					if (bsize == readlen) {
						size_t extra = readExtra(buffer.data() + readlen);
						readlen += extra;
					} else if (bsize != readlen - 4) {
						if (readlen == ntrec*1024) {
							warnx("Assuming buggy 0.4b45 compressed dump. Ignoring compression flag");
							compressed = 0;
							std::memmove(buffer.data()+4, buffer.data(), readlen);
							cflag = 0;
							bsize = readlen;
						} else
							err(1, "Unexpected size of compressed block bsize=%d readlen=%zd clen=%x demangle=%d", bsize, readlen, clen, *demangle_bitfields);
					}
					if (cflag) {
						switch(ctype) {
							case COMPRESS_ZLIB: {
								std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
								std::swap(cmprbuf, buffer);
								uLongf decmprlen = buffer.size();
								int status = uncompress(buffer.data()+4, &decmprlen, cmprbuf.data()+4, bsize);
								switch (status) {
									case Z_OK: break;
									case Z_MEM_ERROR: errx(1, "Memory error while decompressing");
									case Z_BUF_ERROR: errx(1, "BUF error while decompressing");;
									case Z_DATA_ERROR: errx(1, "Corrupt compressed data while decompressing");
									default: errx(1, "Unexpected return from uncompress");
								}
								bsize = decmprlen;
								break;
							}
							case COMPRESS_BZLIB: {
								std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
								std::swap(cmprbuf, buffer);
								unsigned int decmprlen = buffer.size();
								int status = BZ2_bzBuffToBuffDecompress((char*)buffer.data()+4, &decmprlen, (char*)cmprbuf.data()+4, bsize, 0, 0);

								switch (status) {
									case BZ_OK: break;
									case BZ_MEM_ERROR: errx(1, "Memory error while decompressing");
									case BZ_OUTBUFF_FULL: errx(1, "BUF error while decompressing");;
									case BZ_DATA_ERROR:
									case BZ_DATA_ERROR_MAGIC:
									case BZ_UNEXPECTED_EOF:
										errx(1, "Corrupt compressed data while decompressing");
									default: errx(1, "Unexpected return from uncompress");
								}
								bsize = decmprlen;
								break;
							}
							case COMPRESS_LZO: {
								std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
								std::swap(cmprbuf, buffer);
								lzo_uint decmprlen = buffer.size();
								int status = lzo1x_decompress(cmprbuf.data()+4, bsize, buffer.data()+4, &decmprlen, nullptr);

								switch (status) {
									case LZO_E_OK: break;
									case LZO_E_ERROR: errx(1, "LZO error while decompressing");
									case LZO_E_EOF_NOT_FOUND: errx(1, "LZO No EOF error while decompressing");;
									default: errx(1, "Unexpected return from LZO uncompress");
								}
								bsize = decmprlen;
								break;
							}
						}
					}
					bswap_header(buffer.data()+4, bsize, &inctx);
					readlen = bsize + 4;
				} else {
					bswap_header(buffer.data(), readlen, &inctx);
					std::memmove(buffer.data()+4, buffer.data(), readlen);
					readlen += 4;
				}
				/* At this point we have uncompressed data with a compression header (header contains garbage) */
				cflag = 0;
				ctype = 0;
				print_tape(buffer.data(), readlen, &printctx);
			}
		} while(true);
		argv++;
		argc--;
	}

	warnx("Finished");
}

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
