//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file dmparticle.cc
/// \brief Main program of the exoticphysics/dmparticle example

#include "ActionInitialization.hh"
#include "DetectorConstruction.hh"
#include "PhysicsList.hh"

#include "G4RunManagerFactory.hh"
#include "G4Types.hh"
#include "G4UIExecutive.hh"
#include "G4UIcommand.hh"
#include "G4UImanager.hh"
#include "G4VisExecutive.hh"
#include "Randomize.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

int main(int argc, char** argv)
{
  // Instantiate G4UIExecutive if interactive mode
  G4UIExecutive* ui = nullptr;
  if (argc == 1) {
    ui = new G4UIExecutive(argc, argv);
  }

  // choose the Random engine
  CLHEP::HepRandom::setTheEngine(new CLHEP::Ranlux64Engine);

  G4UImanager* UI = G4UImanager::GetUIpointer();
  PhysicsList* phys = new PhysicsList();

  // defined mass of LDM particles
  if (argc > 2) {
    G4String s = argv[2];
    UI->ApplyCommand("/control/verbose 1");
    UI->ApplyCommand("/testex/phys/setLDMPhotonMass " + s + " GeV");
  }
  if (argc > 3) {
    G4String s = argv[3];
    UI->ApplyCommand("/testex/phys/setLDMHiMass " + s + " GeV");
  }

  // Construct the default run manager
  auto* runManager = G4RunManagerFactory::CreateRunManager();
  G4int nThreads = std::min(G4Threading::G4GetNumberOfCores(), 2);
  runManager->SetNumberOfThreads(nThreads);
  G4cout << "===== dmparticle is started with " << runManager->GetNumberOfThreads()
         << " threads =====" << G4endl;

  // set mandatory initialization classes
  DetectorConstruction* det = new DetectorConstruction();
  runManager->SetUserInitialization(phys);
  runManager->SetUserInitialization(det);

  // set user action classes
  runManager->SetUserInitialization(new ActionInitialization(det));

  G4VisManager* visManager = new G4VisExecutive;
  visManager->Initialize();

  if (!ui)  // batch mode
  {
    G4String command = "/control/execute ";
    G4String fileName = argv[1];
    UI->ApplyCommand(command + fileName);
  }
  else  // define visualization and UI terminal for interactive mode
  {
    ui->SessionStart();
    delete ui;
  }
  // job termination
  delete visManager;
  delete runManager;
  return 0;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
