# Copyright 1999-2025 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2

EAPI=8

inherit desktop xdg-utils

DESCRIPTION="Two-way cross-platform file synchronizer"
HOMEPAGE="https://www.seas.upenn.edu/~bcpierce/unison/
	https://github.com/bcpierce00/unison/"

if [[ "${PV}" == *9999* ]] ; then
	inherit git-r3

	EGIT_REPO_URI="https://github.com/bcpierce00/${PN}"
else
	SRC_URI="https://github.com/bcpierce00/${PN}/archive/refs/tags/v${PV}.tar.gz
		-> ${P}.gh.tar.gz"

	KEYWORDS="~amd64 ~arm ~ppc ~x86"
fi

LICENSE="GPL-2"
SLOT="$(ver_cut 1-2)"
IUSE="doc gui +ocamlopt"

BDEPEND="
	dev-lang/ocaml:=[ocamlopt?]
	doc? (
		app-text/dvipsk
		app-text/ghostscript-gpl
		dev-texlive/texlive-latex
	)
"
DEPEND="
	gui? (
		dev-ml/lablgtk:3=[ocamlopt?]
	)
"
RDEPEND="
	dev-lang/ocaml:=
	gui? (
		dev-ml/lablgtk:3=[ocamlopt?]
		|| (
			net-misc/ssh-askpass-fullscreen
			net-misc/x11-ssh-askpass
		)
	)
"
IDEPEND="
	>=app-eselect/eselect-unison-0.4
"

DOCS=( CONTRIBUTING.md README.md src/FEATURES.md src/ROADMAP.txt src/TODO.txt )

QA_FLAGS_IGNORED="usr/bin/${PN}-fsmonitor-${SLOT}"

src_compile() {
	local -a myconf=()

	if ! use ocamlopt ; then
		myconf+=(
			NATIVE="false"
		)
	fi

	# Discard cflags as it will try to pass them to ocamlc...
	emake "${myconf[@]}" CFLAGS="" -j1 tui fsmonitor manpage

	if use gui ; then
		emake "${myconf[@]}" CFLAGS="" -j1 gui
	fi

	if use doc ; then
		VARTEXFONTS="${T}/fonts" emake "${myconf[@]}" CFLAGS="" HEVEA=false docs
	fi
}

src_test() {
	emake test CFLAGS=""
}

src_install() {
	# OCaml generates textrels on 32-bit arches
	if use arm || use ppc || use x86 ; then
		export QA_TEXTRELS='.*'
	fi

	# install manually, since it's just too much
	# work to force the Makefile to do the right thing.
	cd "${S}/src" || die

	local -a bins=(
		unison-fsmonitor
	)

	# Unison GUI is a CLI + GTK GUI in one binary, no need to install both,
	# see bug https://bugs.gentoo.org/941780
	if use gui ; then
		bins+=( unison-gui )
	else
		bins+=( unison )
	fi

	local binname
	for binname in "${bins[@]}" ; do
		exeinto /usr/bin
		newexe "${binname}" "${binname/-gui/}-${SLOT}"
	done

	cd "${S}" || die

	if use gui ; then
		newicon -s scalable icons/U.svg "${PN}-${SLOT}.svg"
		make_desktop_entry "${PN}-${SLOT}" "${PN^} GUI (${SLOT})" "${PN}-${SLOT}"
	fi

	if use doc ; then
		DOCS+=( doc/unison-manual.pdf )
	fi

	doman "man/${PN}.1"
	einstalldocs
}

pkg_postinst() {
	elog "Unison now uses SLOTs, so you can specify servercmd=/usr/bin/unison-${SLOT}"
	elog "in your profile files to access exactly this version over ssh."
	elog "Or you can use 'eselect unison' to set the version."

	eselect unison update

	if use gui ; then
		xdg_icon_cache_update
		xdg_desktop_database_update
	fi
}

pkg_postrm() {
	eselect unison update

	if use gui ; then
		xdg_icon_cache_update
		xdg_desktop_database_update
	fi
}
